/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.handler.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import com.sap.cds.ql.CdsName;
import com.sap.cds.services.Service;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CqnService;

/**
 * Annotation to register a handler for the BEFORE phase of an event on a specific service and entity.
 */
@Target({ ElementType.METHOD })
@Retention(RetentionPolicy.RUNTIME)
public @interface Before {

	/**
	 * Returns the array of events the handler should handle.
	 * In case of an empty array or empty string the event definition is tried to be obtained from the arguments.
	 * For example a {@link CdsCreateEventContext} indicates a {@link CqnService#EVENT_CREATE} event.
	 *
	 * The string '*' is treated as wildcard and matches any event.
	 * If no event definition is provided, either via the annotation or via an argument an error is thrown.
	 * <br>
	 * Defaults to an empty array.
	 *
	 * @return the array of events the handler should handle
	 */
	String[] event() default {};

	/**
	 * Returns the array of entities the handler should handle.
	 * In case of an empty array or empty string the entity definition is tried to be obtained from the arguments.
	 * For example a POJO argument, with {@link CdsName} annotation indicates a certain entity from the model.
	 *
	 * The string '*' is treated as wildcard and matches any entity.
	 * If no entity definition is explicitly provided, it defaults to a wildcard.
	 * <br>
	 * Defaults to an empty array.
	 *
	 * @return the array of entities the handler should handle.
	 */
	String[] entity() default {};

	/**
	 * Returns the names of the services on which the handler should be registered.
	 * In case of an empty array or empty string the registration defaults to the {@link ServiceName} annotation given on class level.
	 *
	 * The string '*' is treated as wildcard and matches any service name.
	 * If no name is given either via this property or via the {@link ServiceName} annotation on class level, registration of the handler fails.
	 * <br>
	 * Defaults to an empty empty array.
	 *
	 * @return the array of names of services on which the handler should be registered.
	 */
	String[] service() default {};

	/**
	 * Returns the array of service types the handler wants to handle.
	 * This can be used together with a service name '*' to register on all services of certain types.
	 * If no type is given, all service types will be considered
	 * <br>
	 * Defaults to an empty array
	 *
	 * @return the array of service types the handler wants to handle
	 */
	Class<? extends Service>[] serviceType() default {};

}
