package com.sap.cds.services.request;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import com.sap.cds.services.CoreFactory;

/**
 * Class that provides information about features of the application that can be enabled or disabled at runtime.
 */
public interface FeatureTogglesInfo {

	/**
	 * Creates a {@link FeatureTogglesInfo} object.
	 * @param featureToggles The feature toggles
	 * @return A {@link FeatureTogglesInfo} object containing all provided feature toggles
	 */
	static FeatureTogglesInfo create(FeatureToggle... featureToggles) {
		return create(Arrays.asList(featureToggles));
	}

	/**
	 * Creates a {@link FeatureTogglesInfo} object.
	 * @param featureToggles A {@link List} of feature toggles
	 * @return A {@link FeatureTogglesInfo} object containing all provided feature toggles in the stream
	 */
	static FeatureTogglesInfo create(List<FeatureToggle> featureToggles) {
		return CoreFactory.INSTANCE.createFeatureTogglesInfo(featureToggles);
	}

	/**
	 * Creates a {@link FeatureTogglesInfo} object.
	 * @param featureToggles A {@link Map} where the key is the feature name and the value indicates if the feature is enable or not
	 * @return A {@link FeatureTogglesInfo} object containing the feature toggles created from the map entries
	 */
	static FeatureTogglesInfo create(Map<String, Boolean> featureToggles) {
		return CoreFactory.INSTANCE.createFeatureTogglesInfo(featureToggles);
	}

	/**
	 * Creates a {@link FeatureTogglesInfo} object.
	 * @return A {@link FeatureTogglesInfo} object containing all feature toggles
	 */
	static FeatureTogglesInfo all() {
		return create(Collections.singletonMap("*", true));
	}

	/**
	 * Creates a {@link FeatureTogglesInfo} object.
	 * @return A {@link FeatureTogglesInfo} object without any toggles
	 */
	static FeatureTogglesInfo none() {
		return create();
	}

	/**
	 * Provides information whether a feature is enabled or not.
	 * @param feature The name of the feature
	 * @return {@code true} if the feature is enabled, {@code false} otherwise
	 */
	boolean isEnabled(String feature);

	/**
	 * @return A {@link Stream} that contains all feature toggles
	 */
	Stream<FeatureToggle> getFeatureToggles();

	/**
	 * @return A {@link Stream} of the currently enabled features
	 */
	Stream<FeatureToggle> getEnabledFeatureToggles();

	/**
	 * @param feature the feature name
	 * @return The {@link FeatureToggle} for the given feature name or {@code null} if not existing
	 */
	FeatureToggle getFeatureToggle(String feature);

}
