package com.sap.cds.services.runtime;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ServiceLoader;

/**
 * Loads implementations of classes via Java's {@link ServiceLoader} and provides
 * the {@link CdsRuntime} to these if requested through the {@link CdsRuntimeAware} interface.
 */
public class ExtendedServiceLoader {

	/**
	 * Loads all implementations of the given class.
	 * The loaded implementations may NOT request the {@link CdsRuntime} through {@link CdsRuntimeAware}.
	 *
	 * @param <T> the type
	 * @param clazz the class
	 * @return the implementations
	 */
	public static <T> Iterator<T> loadAll(Class<T> clazz) {
		return loadAll(clazz, null);
	}

	/**
	 * Loads all implementations of the given class.
	 * The loaded implementations may request the {@link CdsRuntime} through {@link CdsRuntimeAware}.
	 *
	 * @param <T> the type
	 * @param clazz the class
	 * @param runtime the {@link CdsRuntime}
	 * @return the implementations, initialized with the {@link CdsRuntime}
	 */
	public static <T> Iterator<T> loadAll(Class<T> clazz, CdsRuntime runtime) {
		Iterator<T> iterator = ServiceLoader.load(clazz).iterator();
		List<T> runtimeAware = new ArrayList<>();
		iterator.forEachRemaining(i -> {
			if(i instanceof CdsRuntimeAware) {
				if(runtime == null) {
					throw new IllegalStateException("Cannot provide CdsRuntime to implementation for " + clazz.getCanonicalName());
				}
				((CdsRuntimeAware) i).setCdsRuntime(runtime);
			}
			runtimeAware.add(i);
		});
		return runtimeAware.iterator();
	}

	/**
	 * Loads the first implementation of the given class.
	 * The loaded implementation may NOT request the {@link CdsRuntime} through {@link CdsRuntimeAware}.
	 *
	 * @param <T> the type
	 * @param clazz the class
	 * @return the implementation
	 */
	public static <T> T loadSingle(Class<T> clazz) {
		return loadSingle(clazz, null);
	}

	/**
	 * Loads the first implementation of the given class.
	 * The loaded implementation may request the {@link CdsRuntime} through {@link CdsRuntimeAware}.
	 *
	 * @param <T> the type
	 * @param clazz the class
	 * @param runtime the {@link CdsRuntime}
	 * @return the implementation, initialized with the {@link CdsRuntime}
	 */
	public static <T> T loadSingle(Class<T> clazz, CdsRuntime runtime) {
		Iterator<T> iterator = loadAll(clazz, runtime);
		if (iterator.hasNext()) {
			return iterator.next();
		}
		throw new IllegalStateException("Cannot find implementation for " + clazz.getCanonicalName());
	}

}
