/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.runtime;

import java.util.function.Consumer;
import java.util.function.Function;

import com.sap.cds.services.messages.Messages;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.ModifiableParameterInfo;
import com.sap.cds.services.request.ModifiableUserInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;

/**
 * {@link RequestContextRunner} is the main entry to spawn a new {@link RequestContext} with adapted {@link UserInfo} or {@link ParameterInfo}.
 * The current parameters of the calling {@link RequestContext} are not affected.
 */
public interface RequestContextRunner {

	/**
	 * Sets a dedicated {@link UserInfo} for the {@link RequestContext} being opened.
	 *
	 * @param userInfo	The {@link UserInfo}.
	 * @return	this instance
	 */
	RequestContextRunner user(UserInfo userInfo);

	/**
	 * Resets the {@link UserInfo} for the {@link RequestContext} being opened. The empty user is anonymous.
	 *
	 * @return	this instance
	 */
	default RequestContextRunner clearUser() {
		return anonymousUser(); // anonymous user as starting point
	}

	/**
	 * Sets the {@link UserInfo} for the {@link RequestContext} to the anonymous user.
	 * This user is the base-line also used by {@link #clearUser()}.
	 *
	 * @return  this instance
	 */
	RequestContextRunner anonymousUser();

	/**
	 * Sets the {@link UserInfo} for the {@link RequestContext} as provided by the {@link UserInfoProvider}.
	 * By default this means the {@link UserInfo} is resolved from the HTTP request.
	 * You may override the default provider during runtime configuration phase by means of {@link CdsRuntimeConfigurer#provider(UserInfoProvider)}.
	 *
	 * @return	this instance
	 */
	RequestContextRunner providedUser();

	/**
	 * Sets the {@link UserInfo} for the {@link RequestContext} to the system user of the provider account. This is equivalent to
	 * calling {@code systemUser(null)}
	 *
	 * @return  this instance
	 */
	RequestContextRunner systemUserProvider();

	/**
	 * Sets the {@link UserInfo} for the {@link RequestContext} to the system user. The tenant will be automatically propagated
	 * from the current {@link UserInfo}. This is equivalent to calling {@code systemUser(userInfo.getTenant())}.
	 *
	 * @return  this instance
	 */
	RequestContextRunner systemUser();

	/**
	 * Sets the {@link UserInfo} for the {@link RequestContext} to the system user of the specified subscriber account.
	 *
	 * @param tenantId Tenant in which the {@link RequestContext} will be executed
	 * @return  this instance
	 */
	RequestContextRunner systemUser(String tenantId);

	/**
	 * Sets the {@link UserInfo} for the {@link RequestContext} to the internal privileged user who passes all authorization checks.
	 *
	 * @return	this instance
	 */
	RequestContextRunner privilegedUser();

	/**
	 * Opens a {@link RequestContext} with an adapted {@link UserInfo}.
	 * The passed {@link ModifiableUserInfo} API can be used to define modifications which are valid within the scope of the new {@link RequestContext}.
	 * <p>
	 * The modifications are done on the {@link UserInfo} of the outer {@link RequestContext} if existing.
	 * Otherwise they are done on the provided {@link UserInfo} from {@link UserInfoProvider}.
	 *
	 * @param contextUser	the {@link ModifiableUserInfo} as modification layer on top of current {@link UserInfo}
	 * @return	this instance
	 */
	RequestContextRunner modifyUser(Consumer<ModifiableUserInfo> contextUser);

	/**
	 * Sets a dedicated {@link ParameterInfo} for the {@link RequestContext} being opened.
	 *
	 * @param parameterInfo	the {@link ParameterInfo}.
	 * @return	this instance
	 */
	RequestContextRunner parameters(ParameterInfo parameterInfo);

	/**
	 * Sets a dedicated {@link FeatureTogglesInfo} for the {@link RequestContext} being opened.
	 * Throws an exception in case the {@code RequestContext} being opened has a parent (i.e. is nested).
	 *
	 * @param featureTogglesInfo	the {@link FeatureTogglesInfo}.
	 * @return	this instance
	 */
	RequestContextRunner featureToggles(FeatureTogglesInfo featureTogglesInfo);

	/**
	 * Resets the {@link ParameterInfo} for the {@link RequestContext} being opened.
	 *
	 * @return	this instance
	 */
	RequestContextRunner clearParameters();

	/**
	 * Sets the {@link ParameterInfo} for the {@link RequestContext} as provided by the {@link ParameterInfoProvider}.
	 * By default this means the {@link ParameterInfo} is resolved from the HTTP request.
	 * You may override the default provider during runtime configuration phase by means of {@link CdsRuntimeConfigurer#provider(ParameterInfoProvider)}.
	 *
	 * @return	this instance
	 */
	RequestContextRunner providedParameters();

	/**
	 * Opens a {@link RequestContext} with an adapted {@link ParameterInfo}.
	 * The passed {@link ModifiableParameterInfo} API can be used to define modifications which are valid within the scope of the new {@link RequestContext}.
	 * <p>
	 * The modifications are done on the {@link ParameterInfo} of the outer {@link RequestContext} if existing.
	 * Otherwise they are done on the provided {@link ParameterInfo} from {@link ParameterInfoProvider}.
	 *
	 * @param contextParameters	the {@link ModifiableParameterInfo} as modification layer on top of current {@link ParameterInfo}
	 * @return	this instance
	 */
	RequestContextRunner modifyParameters(Consumer<ModifiableParameterInfo> contextParameters);

	/**
	 * Resets the {@link Messages} for the {@link RequestContext} being opened.
	 *
	 * @return  this instance
	 */
	RequestContextRunner clearMessages();

	/**
	 * Opens a {@link RequestContext} and runs the given handler within its scope.
	 * The {@code RequestContext} will be propagated to all
	 * {@link com.sap.cds.services.EventContext} instances in the service call hierarchy.
	 *
	 * @param <T>	The type of the response
	 * @param requestHandler The handler for processing the request within the context
	 * @return A generic response object of the handler
	 */
	<T> T run(Function<RequestContext, T> requestHandler);

	/**
	 * Opens a {@link RequestContext} and runs the given handler within its scope.
	 * The {@code RequestContext} will be propagated to all {@link com.sap.cds.services.EventContext} instances in the service call hierarchy.
	 * <p>
	 * If {@link UserInfo} has not been specified explicitly, it is taken from the outer {@link RequestContext} or, if there is no outer context, from {@link UserInfoProvider}.
	 * The same holds for {@link ParameterInfo}.
	 *
	 * @param requestHandler The handler for processing the request within the context
	 */
	void run(Consumer<RequestContext> requestHandler);
}
