/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.application;

import java.util.List;

import com.sap.cds.CdsData;
import com.sap.cds.Struct;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.EventName;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.messages.Message;

/**
 * {@link EventContext} for the event {@link ApplicationLifecycleService#EVENT_ERROR_RESPONSE}, that
 * allows to handle transformation of exceptions and messages to an error response
 */
@EventName(ApplicationLifecycleService.EVENT_ERROR_RESPONSE)
public interface ErrorResponseEventContext extends EventContext {

	/**
	 * Creates an {@link EventContext} already overlayed with this interface. The event is set to be {@link ApplicationLifecycleService#EVENT_ERROR_RESPONSE}
	 * @return the {@link ErrorResponseEventContext}
	 */
	static ErrorResponseEventContext create() {
		return EventContext.create(ErrorResponseEventContext.class, null);
	}

	/**
	 * Sets the {@link ServiceException} that should be transformed into an error response.
	 * @param exception the {@link ServiceException} that should be transformed into an error response.
	 */
	void setException(ServiceException exception);

	/**
	 * Returns the {@link ServiceException} that caused the error being transformed into an error response.
	 * @return the {@link ServiceException} that should be transformed into an error response.
	 */
	ServiceException getException();

	/**
	 * Sets the {@link ErrorResponse} with messages and target HTTP status.
	 * @param errorResponse the {@link ErrorResponse} with messages and target HTTP status.
	 */
	void setResult(ErrorResponse errorResponse);

	/**
	 * Returns the {@link ErrorResponse} with messages and target HTTP status.
	 * @return the {@link ErrorResponse} with messages and target HTTP status.
	 */
	ErrorResponse getResult();

	/**
	 *  Error response generated by the event handler.
	 */
	interface ErrorResponse extends CdsData {

		/**
		 * Creates a new instance of {@link ErrorResponse}.
		 * @return the instance of {@link ErrorResponse}.
		 */
		static ErrorResponse create() {
			return Struct.create(ErrorResponse.class);
		}

		/**
		 * Returns the HTTP status to be used in the error response.
		 * @return the HTTP status.
		 */
		Integer getHttpStatus();

		/**
		 * Sets the HTTP status to be used in the error response.
		 * @param httpStatus the HTTP status to be used in the error response.
		 */
		void setHttpStatus(Integer httpStatus);

		/**
		 * Returns the list of {@link Message} objects to be written in the error response.
		 * @return the list of {@link Message} objects to be written in the error response.
		 */
		List<Message> getMessages();

		/**
		 * Sets the list of {@link Message} objects to be written in the error response.
		 * The first message of the list will be used as the primary error in the response generated by the OData V2 and V4 protocol adapters.
		 * @param messages the list of {@link Message} objects to be written in the error response.
		 */
		void setMessages(List<Message> messages);

	}
}
