/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.environment;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.sap.cds.docs.config.DocumentedProperty;
import com.sap.cds.services.environment.CdsProperties.Composite.CompositeServiceConfig.Route;
import com.sap.cds.services.outbox.OutboxService;

/**
 * Properties that control the CDS runtime
 */
@lombok.Getter
@lombok.Setter
public class CdsProperties {

	/** Properties for environments, like local development or Cloud Foundry. */
	private Environment environment = new Environment();
	/** Properties for the primary data source, used by the default persistence service. */
	private DataSource dataSource = new DataSource();
	/** Properties for the CDS model. */
	private Model model = new Model();
	/** Properties for security configurations of services and endpoints. */
	private Security security = new Security();
	/** Properties for the index page. */
	private Servlet indexPage = new Servlet("/", true);
	/** Properties for the OData V4 protocol adapter. */
	private ODataV4 odataV4 = new ODataV4();
	/** Properties for the OData V2 protocol adapter. */
	private ODataV2 odataV2 = new ODataV2();
	/** Properties for the HCQL protocol adapter. */
	private Hcql hcql = new Hcql();
	/** Properties for messaging services. */
	private Messaging messaging = new Messaging();
	/** Properties for multitenancy and extensibility. */
	private MultiTenancy multiTenancy = new MultiTenancy();
	/** Properties for persistence services. */
	private Persistence persistence = new Persistence();
	/** Properties for application services. */
	private Application application = new Application();
	/** Properties for remote services. */
	private Remote remote = new Remote();
	/** Properties for locale configurations. */
	private Locales locales = new Locales();
	/** Properties for error handling. */
	private Errors errors = new Errors();
	/** Properties for draft-enabled entities. */
	private Drafts drafts = new Drafts();
	/** Properties for augmentation of CQN queries. */
	private Query query = new Query();
	/** Properties for SQL generation.  **/
	private Sql sql = new Sql();
	/** Properties for AuditLog configuration. **/
	private AuditLog auditLog = new AuditLog();
	/** Properties for Outbox configuration. **/
	private Outbox outbox = new Outbox();
	/** Properties for Cloud SDK integration configuration. */
	@DocumentedProperty(false)
	private CloudSdk cloudSdk = new CloudSdk();

	@lombok.Getter
	@lombok.Setter
	public static class Environment {
		/** Properties for the local environment. */
		private Local local = new Local();
		/** Properties for the production environment. */
		private Production production = new Production();
		/** Properties for specifying a command environment. No web server or background activities are initiated. */
		@DocumentedProperty(false)
		private Enabled command = new Enabled(false);
		/** Properties for Kubernetes. */
		@Deprecated
		@DocumentedProperty(false)
		private K8s k8s = new K8s();

		@lombok.Getter
		@lombok.Setter
		public static class Local {

			/**
			 * Specification of the default environment JSON file. If this property starts with "classpath:", it is read
			 * as classloader resource. Otherwise it is interpreted as file-system path.
			 * The file content follows the structure of Cloud Foundry's VCAP_SERVICES and VCAP_APPLICATION environment variables.
			 * If this property specifies a folder, the filename `default-env.json` is appended to it.
			 */
			private String defaultEnvPath;

		}

		@lombok.Getter
		@lombok.Setter
		public static class Production extends Enabled {

			/** The Spring Boot profile to attach production-optimized default property values to. */
			private String profile = "cloud";

			public Production() {
				super(null);
			}

		}

		@lombok.Getter
		@lombok.Setter
		public static class K8s {
			/**
			 * Secrets directory where service bindings are found according to the following conventions:
			 * `[secretsPath]/[serviceName]/[serviceInstanceName]`
			 * `[serviceName]` and `[serviceInstanceName]` are directories with the name of the service and service instance respectively.
			 * The directory `[serviceInstanceName]` must contain the service binding credentials using files with its name being the key and its content being the value.
			 */
			private String secretsPath = "/etc/secrets/sapcp";
			/**
			 * Properties to explicitly configure service bindings with an arbitrary secrets path.
			 * The metadata for these service bindings can be fully specified, to ensure CAP is able detect the binding in auto-configuration.
			 * The key can be chosen arbitrarily and is used as the service binding name, if the `name` property is not explicitly defined.
			 */
			private Map<String, ServiceBindingConfig> serviceBindings = new HashMap<>();

			public Map<String, ServiceBindingConfig> getServiceBindings() {
				serviceBindings.forEach((k, v) -> {
					if(v.getName() == null || v.getName().trim().isEmpty()) {
						v.setName(k);
					}
				});
				return serviceBindings;
			}

			@lombok.Getter
			@lombok.Setter
			public static class ServiceBindingConfig {
				/** The name of the service binding. */
				private String name;
				/** The path to the secrets directory. */
				private String secretsPath;
				/** The name of the service. */
				private String service;
				/** The optional name of the service plan. */
				private String plan;
				/** The optional list of service tags. */
				private List<String> tags;
			}
		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class DataSource {

		/** Properties for CSV initialization. */
		private Csv csv = new Csv();
		/** Properties to control the `DataSource` auto-configuration from service bindings. */
		private Enabled autoConfig = new Enabled(true);
		/** Determines, if the data source is considered embedded (in-memory). */
		private boolean embedded;
		/** The name of the primary service binding, used by the default persistence service. */
		private String binding;


		@lombok.Getter
		@lombok.Setter
		public static class Csv {

			/**
			 * Determines in which scenarios the default persistence service is initialized with CSV data.
			 * By default, CSV initialization only happens, if the data source is embedded (in-memory).
			 * Possible values are: `embedded`, `never`, `always`.
			 */
			private String initializationMode = "embedded"; // or 'never' or 'always'
			/** The file suffix of CSV files. */
			private String fileSuffix = ".csv";
			/**
			 * The file-system paths to search for CSV files in.
			 * Using `/**` at the end of the path triggers a recursive search.
			 */
			private List<String> paths = Arrays.asList("db/data/**", "db/csv/**", "../db/data/**", "../db/csv/**");
			/**
			 * Enables import of all CSV files in a single changeset. By default, each CSV file is imported in a separate changeset.
			 */
			private boolean singleChangeset;

		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Model {

		/** The resource path to the `csn.json` file. */
		private String csnPath = "edmx/csn.json";
		/** Properties for usage of Universal CSN. */
		@DocumentedProperty(false)
		private Enabled universalCsn = new Enabled(false);
		/** The model provider configuration. */
		private Provider provider = new Provider();
		/** Determines, if UI annotations are included in the model. */
		private boolean includeUiAnnotations = false;

		@lombok.Getter
		@lombok.Setter
		public static class Provider {
			/** The URL of the Model Provider. */
			private String url;
			/** Assume that dynamic models contain tenant-specific extensions. */
			private boolean extensibility = true;
			/** Assume that dynamic models are feature-toggle specific. */
			private boolean toggles = true;
			/** Properties for the CDS model and EDMX metadata caches. */
			private Cache cache = new Cache();

			@lombok.Getter
			@lombok.Setter
			public static class Cache {
				/** The number of entries in the CDS model and EDMX metadata caches. */
				private int maxSize = 20;
				/** The lifetime of an entry in seconds after the entry's creation, the most recent replacement of its value, or its last access. */
				private int expirationTime = 600;
				/** The time in seconds after which a cached entry is refreshed. */
				private int refreshTime = 60;
			}
		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class Security {

		/** Determines, if potentially sensitive data, for example values in CQN queries, might be logged. */
		private boolean logPotentiallySensitive = false;

		/** Properties for authentication in general. */
		private Authentication authentication = new Authentication();
		/** Properties for authorization. */
		private Authorization authorization = new Authorization();
		/** Properties for authentication based on Identity Service (IAS). */
		private Identity identity = new Identity();
		/** Properties for authentication based on XSUAA. */
		private Xsuaa xsuaa = new Xsuaa();
		/** Properties for authentication based on mock-users. */
		private Mock mock = new Mock();

		@lombok.Getter
		@lombok.Setter
		public static class Authentication {

			/**
			 * Determines the mode that is used when authenticating endpoints of protocol-adapters. Possible values are:
			 * - never: No endpoint requires authentication
			 * - model-relaxed: Authentication is derived from @requires/@restrict, defaults to public endpoints.
			 * - model-strict: Authentication is derived from @requires/@restrict, defaults to authenticated endpoints.
			 * - always: All endpoints require authentication
			 */
			private String mode = "model-strict";
			/** Determines, if OData $metadata endpoints enforce authentication. */
			private boolean authenticateMetadataEndpoints = true;
			/** Determines, if security configurations enforce authentication for endpoints not managed by protocol-adapters. */
			private boolean authenticateUnknownEndpoints = true;
			/** Determines, if the provider tenant is normalized to `null`. */
			private boolean normalizeProviderTenant = true;

			/** Properties for the Spring security auto-configuration. */
			private Enabled authConfig = new Enabled(true);
		}

		@lombok.Getter
		@lombok.Setter
		public static class Authorization extends Enabled {

			/**
			 * Properties to control the protection of drafts.
			 * If a draft is protected, it is only accessible by its creator.
			 */
			private Enabled draftProtection = new Enabled(true);
			/** Controls the default behavior of empty attributes in a filter condition of instance-based authorization (where-clause of @restrict). */
			@Deprecated
			@DocumentedProperty(false)
			private boolean emptyAttributeValuesAreRestricted = true;

			/**
			 * Determines, if the deep authorization checks are enforced
			 */
			@DocumentedProperty(false)
			private Enabled deep = new Enabled(true);

			@DocumentedProperty(false)
			private InstanceBased instanceBased = new InstanceBased();

			public Authorization() {
				super(true);
			}

			@lombok.Getter
			@lombok.Setter
			public static class InstanceBased {

				/**
				 * Enables instance-based auth for custom events.
				 */
				@DocumentedProperty(false)
				private Enabled customEvents = new Enabled(true);
			}

		}

		@lombok.Getter
		@lombok.Setter
		public static class Identity extends Enabled {

			/** The name of the IAS service binding, used for the identity security auto-configuration. */
			private String binding;
			/** Determines, if service plans are mapped to roles in the UserInfo object. */
			@DocumentedProperty(false)
			private boolean exposePlansAsRoles = true;

			public Identity() {
				super(true);
			}
		}

		@lombok.Getter
		@lombok.Setter
		public static class Xsuaa extends Enabled {

			/** The name of the XSUAA service binding, used for the XSUAA security auto-configuration. */
			private String binding;

			public Xsuaa() {
				super(true);
			}
		}

		@lombok.Getter
		@lombok.Setter
		public static class Mock extends Enabled {

			/**
			 * The mock-users, used for basic authentication in local development and test scenarios.
			 * The key can be chosen arbitrarily and is used as the user name, if the `name` property is not explicitly defined.
			 * In addition it can be leveraged to split configuration across multiple profiles.
			 */
			private Map<String, User> users = new HashMap<>();

			/** Enables a list of default mock-users. */
			private Enabled defaultUsers = new Enabled(true);

			/**
			 * The tenants in local development and test scenarios.
			 * The key can be chosen arbitrarily and is used as the tenant name, if the `name` property is not explicitly defined.
			 * In addition it can be leveraged to split configuration across multiple profiles.
			 */
			private Map<String, Tenant> tenants = new HashMap<>();

			public Mock() {
				super(true);
			}

			@lombok.Getter
			@lombok.Setter
			public static class User {

				/** The ID of the mock-user. */
				private String id;
				/**
				 * The (mandatory) name of the mock-user.
				 * It's used to perform the basic authentication.
				 */
				private String name; // mandatory
				/**
				 * The (optional) password of the mock-user.
				 * It's used to perform the basic authentication.
				 */
				private String password = ""; // empty password is default
				/** The tenant of the mock-user. */
				private String tenant; // optional
				/** Determines, if this mock-user is treated as a system user. */
				private boolean systemUser = false;
				/** Determines, if this mock-user is treated as the privileged user. */
				private boolean privileged = false;
				/** Determines, if this mock-user is treated as the internal user. */
				private boolean internalUser = false;
				/** The list of roles, that are assigned to this mock-user. */
				private List<String> roles = new ArrayList<>();
				/** The list of enabled feature toggles for this user. If set, it overrules features of the tenant (if provided). */
				private List<String> features;

				/**
				 * A map of user attributes, that are assigned to the mock-user.
				 * The name of the attribute needs to be given as the key.
				 * The attribute values are provided as a list.
				 */
				private Map<String, List<String>> attributes = new HashMap<>();
				/**
				 * A map of additional properties of the mock-user.
				 * It can be used to mock authentication specific properties (e.g. email address).
				 * The name of the additional attribute needs to be given as the key.
				 * The value of the attribute can be provided as an arbitrary object.
				 */
				private Map<String, Object> additional = new HashMap<>();

				public boolean isValid() {
					return getName() != null && !getName().isEmpty() && getPassword() != null;
				}

			}

			public Map<String, User> getUsers() {
				users.forEach((k, v) -> {
					if(v.getName() == null || v.getName().trim().isEmpty()) {
						v.setName(k);
					}
				});
				if (defaultUsers.isEnabled()) {
					return withDefaultUsers(users);
				} else {
					return users;
				}
			}

			private Map<String, User> withDefaultUsers(Map<String, User> configured) {
				Map<String, User> users = new HashMap<>(configured);
				User authenticated = new User();
				authenticated.setName("authenticated");
				users.put("authenticated", authenticated);

				User system = new User();
				system.setName("system");
				system.setSystemUser(true);
				users.put("system", system);

				User internal = new User();
				internal.setName("internal");
				internal.setInternalUser(true);
				users.put("internal", internal);

				User privileged = new User();
				privileged.setName("privileged");
				privileged.setPrivileged(true);
				users.put("privileged", privileged);

				return users;
			}

			@lombok.Getter
			@lombok.Setter
			public static class Tenant {
				/**
				 * The (mandatory) name of the tenant
				 */
				private String name; // mandatory
				/** The list of enabled feature toggles for this tenant. */
				private List<String> features = new ArrayList<>();
			}

			public Map<String, Tenant> getTenants() {
				tenants.forEach((k, v) -> {
					if(v.getName() == null || v.getName().trim().isEmpty()) {
						v.setName(k);
					}
				});
				return tenants;
			}
		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class ODataV4 {

		/** Properties of the OData V4 protocol adapter endpoint. */
		private Servlet endpoint = new Servlet("/odata/v4", true);
		/** Determines, if URLs in the @odata.context response annotation are absolute. */
		private boolean contextAbsoluteUrl = false;
		/** The JAR resource path to search for OData V4 EDMX files. */
		private String edmxPath = "edmx/v4";
		/** Determines, if localization is applied to the EDMX lazily, saving memory resources. */
		private Enabled lazyI18n = new Enabled(true);
		@Deprecated
		@DocumentedProperty(false)
		private Serializer serializer = new Serializer();
		/** Properties for EDM model creation, based on the CdsModel reflection API. */
		@DocumentedProperty(false)
		private Enabled cdsToEdm = new Enabled(false);
		/** Properties for OData batch requests. */
		private Batch batch = new Batch();
		/** Properties for handling of $apply query parameters. */
		@DocumentedProperty(false)
		private Apply apply = new Apply();
		/**
		 * Defines the handling of the $search query option.
		 * In `odata-strict` mode the search string is parsed and interpreted according to the OData v4 $search specification.
		 * In `pass-through` mode the search string is passed through to the data store.
		 * In `odata-lenient` mode the search string is parsed and only passed through on parsing errors.
		 */
		private String searchMode = "odata-lenient";

		@lombok.Getter
		@lombok.Setter
		public static class Serializer {
			/**
			 * If set to true, switches to buffered serialization mode, allowing to catch
			 * serialization exceptions and set corresponding error headers before
			 * sending the response to the client.
			 * To be used by those clients, which don't support direct streaming.
			 */
			private boolean buffered = false;

		}

		@lombok.Getter
		@lombok.Setter
		public static class Batch {

			/** Defines the maximum number of requests within OData batch requests. By default, no limit applies. */
			private long maxRequests = -1;

		}

		@lombok.Getter
		@lombok.Setter
		public static class Apply {
			/** Indicates that $apply is captured in CQN transformation objects and can be handled in custom code. */
			private Enabled transformations = new Enabled(false);
		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class ODataV2 {

		/** Properties of the OData V2 protocol adapter endpoint. */
		private Servlet endpoint = new Servlet("/odata/v2", true);
		/** The JAR resource path to search for OData V2 EDMX files. */
		private String edmxPath = "edmx/v2";
		/**
		 * Determines whether OData functions 'substringof', 'startswith' and 'endswith' are
		 * case-sensitive.
		 */
		private boolean caseSensitiveFilter = true;
		/**
		 * Defines the handling of the $search query option.
		 * In `odata-strict` mode the search string is parsed and interpreted according to the OData v4 $search specification.
		 * In `pass-through` mode the search string is passed through to the data store.
		 * In `odata-lenient` mode the search string is parsed and only passed through on parsing errors.
		 */
		private String searchMode = "odata-lenient";
		/**
		 * Properties for OData batch requests.
		 */
		private Batch batch = new Batch();

		@lombok.Getter
		@lombok.Setter
		public static class Batch {

			/**
			 * Defines the maximum number of requests within OData batch requests. By default, no limit applies.
			 */
			private long maxRequests = -1;

		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Hcql {

		/** Properties of the HCQL protocol adapter endpoint. */
		private Servlet endpoint = new Servlet("/hcql", true);

	}

	@lombok.Getter
	@lombok.Setter
	public static class Composite {

		/**
		 * Properties for composite services.
		 * The key can be chosen arbitrarily and is used as the composite service name, if the `name` property is not explicitly defined.
		 * In addition it can be leveraged to split configuration across multiple profiles.
		 */
		private Map<String, CompositeServiceConfig> services = new HashMap<>();

		public Map<String, CompositeServiceConfig> getServices() {
			services.forEach((k, v) -> {
				if(v.getName() == null || v.getName().trim().isEmpty()) {
					v.setName(k);
				}
			});
			return services;
		}

		@lombok.Getter
		@lombok.Setter
		public static class CompositeServiceConfig {

			/** The name of the composite service. */
			private String name;
			/**
			 * The list of routes of the composite service.
			 * The first route that matches is used. Therefore the order of these routes has significance.
			 */
			private List<Route> routes = new ArrayList<>();

			public CompositeServiceConfig(String name) {
				this.name = name;
			}

			public CompositeServiceConfig() {
				// empty for constructing from YAML
			}

			@lombok.Getter
			@lombok.Setter
			public static class Route {

				/** The target service of the route. */
				private String service;
				/** The list of events/topics, which are propagated to/from the target service. */
				private List<String> events = new ArrayList<>();

			}
		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class Messaging {

		/** Properties for receiving messages through the webhook. */
		private MessagingWebhooks webhooks = new MessagingWebhooks();
		/** Determines, if queues are deleted and recreated during startup of the application (only for development). */
		@DocumentedProperty(false)
		private boolean resetQueues = false;

		/**
		 * The list of routes for the composite messaging service.
		 * The first route that matches is used. Therefore the order of these routes has significance.
		 */
		private List<Route> routes = new ArrayList<>();
		/**
		 * Properties for messaging services.
		 * The key can be chosen arbitrarily and is used as the messaging service name, if the `name` property is not explicitly defined.
		 * In addition it can be leveraged to split configuration across multiple profiles.
		 */
		private Map<String, MessagingServiceConfig> services = new HashMap<>();


		public Map<String, MessagingServiceConfig> getServices() {
			services.forEach((k, v) -> {
				if(v.getName() == null || v.getName().trim().isEmpty()) {
					v.setName(k);
				}
			});
			return services;
		}

		public MessagingServiceConfig getService(String name) {
			return getServices().values().stream()
					.filter(s -> Objects.equals(s.getName(), name))
					.findFirst()
					.orElse(new MessagingServiceConfig(name));
		}

		public List<MessagingServiceConfig> getServicesByKind(String kind) {
			return getServices().values().stream()
					.filter(s -> Objects.equals(s.getKind(), kind))
					.collect(Collectors.toList());
		}

		public List<MessagingServiceConfig> getServicesByBinding(String binding) {
			return getServices().values().stream()
					.filter(s -> (Objects.equals(s.getBinding(), binding) || (s.getBinding() == null && Objects.equals(s.getName(), binding))))
					.collect(Collectors.toList());
		}

		@lombok.Getter
		@lombok.Setter
		public static class MessagingWebhooks extends Enabled {

			/** The application webhook URL. This URL is used when registering the webhook at the message broker. */
			private String url;

			public MessagingWebhooks() {
				super(true);
			}
		}

		@lombok.Getter
		@lombok.Setter
		public static class MessagingServiceConfig extends Enabled {

			/** The name of the messaging service. */
			private String name;
			/**
			 * The kind of the messaging service.
			 * It usually reflects the corresponding service binding type.
			 * Possible values are: `local-messaging`, `file-based-messaging`,
			 * `enterprise-messaging`, `message-queuing`, `redis-pubsub`,
			 * `kafka-channel-messaging`.
			 */
			private String kind;
			/**
			 * The name of the service binding used for this messaging service.
			 * In case of file-based-messaging this specifies the file-system path to the exchange file.
			 */
			private String binding;
			/** The string used to prefix topics when subscribing to events. */
			private String subscribePrefix;
			/** The string used to prefix topics when publishing events. */
			private String publishPrefix;
			/**
			 * The message format to be assumed on subscriptions and applied when publishing.
			 * Possible values are: `cloudevents`
			 */
			private String format;
			/**
			 * Determines, if messages are represented as a plain String (`false`)
			 * or always structured as two separate maps, representing data and headers (`true`).
			 * Setting this property has effects on how the messages are represented in the broker.
			 */
			@Deprecated(forRemoval = true, since = "3.0.0")
			private boolean structured = true;
			/** Properties for the JMS client connection. */
			private Connection connection = new Connection();
			/** Properties of the queue that is created for the messaging service. */
			private Queue queue = new Queue();
			/** Properties to control, if and how the Outbox should be used for this messaging service. */
			private Outbox outbox = new Outbox();

			public MessagingServiceConfig() {
				super(true);
			}

			public MessagingServiceConfig(String name) {
				super(true);
				this.name = name;
			}

			@lombok.Getter
			@lombok.Setter
			public static class Connection {

				/**
				 * Determines, if this messaging service uses its own dedicated JMS client connection.
				 * By default, JMS client connections to the same messaging broker are shared.
				 */
				private boolean dedicated = false;
				/**
				 * Properties passed to the JMS client connection.
				 * The possible keys and values depend on the messaging service implementation.
				 */
				private Map<String, String> properties = new HashMap<>();

				/**
				 * Connection pool properties for the rest client
				 */
				@DocumentedProperty(false)
				private ConnectionPool connectionPool = new ConnectionPool();
			}

			@lombok.Getter
			@lombok.Setter
			public static class Queue {

				/**
				 * The name of the queue.
				 * The queue may already exist with some custom configuration. In that case the queue is not recreated.
				 */
				private String name;
				/**
				 * Properties passed to the messaging broker when creating the queue.
				 * The possible keys and values depend on the messaging service implementation.
				 */
				private Map<String, Object> config = new HashMap<>();
				/**
				 * A list of additional topics, that are subscribed on the queue.
				 * By default event handler registrations should be used to trigger subscriptions.
				 * This property is intended for purposes when subscriptions can not be inferred from event handler registrations.
				 */
				private List<String> subscriptions = new ArrayList<>();
				/**
				 * Specifies whether a queue listener should be connected even if no subscription is available.
				 */
				private boolean forceListening = false;

			}

			@lombok.Getter
			@lombok.Setter
			public static class Outbox extends Enabled {

				/** The name of the outbox service to use. If not available or set, the in-memory outbox is used as fallback. */
				private String name = OutboxService.PERSISTENT_ORDERED_NAME;

				public Outbox() {
					super(true);
				}
			}
		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class MultiTenancy {

		/** Properties of the subscription HTTP endpoints. */
		private Servlet endpoint = new Servlet("/mt/v1.0/subscriptions", true);
		/** Properties for the subscription manager service. */
		private SubscriptionManager subscriptionManager = new SubscriptionManager();
		/** Properties for the multitenant aware datasource. */
		private DataSource dataSource = new DataSource();
		/** Properties for the service-manager client. */
		private ServiceManager serviceManager = new ServiceManager();
		/** Properties for authorization. */
		private Security security = new Security();
		@Deprecated
		@DocumentedProperty(false)
		private Deployer deployer = new Deployer();
		/** Properties for the MTX sidecar client. */
		private Sidecar sidecar = new Sidecar();
		/** Properties for the URL to the application's UI endpoints. */
		private AppUi appUi = new AppUi();
		/** Properties for health check of the multitenant-aware datasource. */
		private HealthCheck healthCheck = new HealthCheck();
		/** Properties for DB lifecycle management via Liquibase. */
		private Liquibase liquibase = new Liquibase();
		/** Properties for the ProvisioningService from @sap/cds-mtxs. */
		private ProvisioningService provisioning = new ProvisioningService();
		/** Properties for the SQLite-based MTX mock for local development and testing. */
		private Mock mock = new Mock();
		/** Package to be included in the component scan of the Subscribe, Deploy and Unsubscribe main methods.  */
		private String componentScan;
		/** Properties for automatically reported subscription dependencies. */
		private Dependencies dependencies = new Dependencies();

		@lombok.Getter
		@lombok.Setter
		public static class Dependencies {
			/** Indicates if a subscription dependency should be registered for a bound service instance of Destination service. */
			private boolean destination = false;
		}

		@lombok.Getter
		@lombok.Setter
		public static class SubscriptionManager {
			/** Properties of the subscription HTTP endpoint for Subscription Manager Service (SMS). */
			private Servlet endpoint = new Servlet("/mt/sms/subscriptions", true);
			/** Header name for forwarded client certificates. */
			private String clientCertificateHeader = "X-Forwarded-Client-Cert";
		}

		@lombok.Getter
		@lombok.Setter
		public static class DataSource {

			/**
			 * Pool to use for the multitenant-aware datasource.
			 * Possible values are: `hikari`, `tomcat`.
			 */
			private String pool = "hikari";
			/**
			 * Properties to control how the connection pools are maintained.
			 * This allows to configure, that the connection pools for tenants contained in the same database are combined. Instead of having a dedicated connection pool for each tenant schema.
			 */
			private Enabled combinePools = new Enabled(false);
			/** Optional tenant encryption mode used by the HANA database. Possible values are DEDICATED_KEY and MANAGED_KEY.  */
			private String hanaEncryptionMode;

		}

		@lombok.Getter
		@lombok.Setter
		public static class ServiceManager {

			/** The cache refresh interval (as Duration). */
			private Duration cacheRefreshInterval = Duration.ofMinutes(2);
		}

		@lombok.Getter
		@lombok.Setter
		public static class Security {

			/** The scope by which the subscription endpoints are authorized. */
			private String subscriptionScope = "mtcallback";

		}

		@lombok.Getter
		@lombok.Setter
		public static class Deployer {

			private String url;
			private String user;
			private String password;
			private Duration asyncTimeout = Duration.ofMinutes(10);

		}

		@lombok.Getter
		@lombok.Setter
		public static class Sidecar {
			/**
			 * The URL of the MTX sidecar.
			 * Setting this property, in combination with a present service-manager service binding, activates the MTX features.
			 */
			private String url;

			/** Specifies the time interval between polling the status of the lifecycle operation. */
			@DocumentedProperty(false)
			private Duration pollingInterval = Duration.ofSeconds(5);
			/** Specifies the maximum waiting time for the operation to finish. */
			private Duration pollingTimeout = Duration.ofMinutes(20);
		}

		@lombok.Getter
		@lombok.Setter
		public static class AppUi {

			/** The URL to the application's UI, used for the 'Go to Application' link. */
			private String url;
			/**
			 * The separator for the tenant in the URL.
			 * Possible values are: `.`, `-`.
			 */
			private String tenantSeparator;

		}

		@lombok.Getter
		@lombok.Setter
		public static class HealthCheck extends Enabled {

			/** The statement that is used when executing a health check of the multitenant-aware datasource. */
			private String healthCheckStatement = ""; // empty means chosen by mt library
			/** The time a health check result is cached and no further health checks are performed. */
			private Duration interval = Duration.ofSeconds(10);

			public HealthCheck() {
				super(true);
			}

		}

		@lombok.Getter
		@lombok.Setter
		public static class Liquibase {

			/** The location of the master Liquibase file. */
			private String changeLog = "/db/changelog/db.changelog-master.yaml";
			/** Optional: Comma separated list of active contexts. */
			private String contexts;

		}

		@lombok.Getter
		@lombok.Setter
		public static class ProvisioningService {
			/** URL of the provisioning service. If not specified, `cds.multitenancy.sidecar.url` is taken as default. */
			private String url;
			/** Specifies the time interval between polling the status of the provisioning operation. */
			@DocumentedProperty(false)
			private Duration pollingInterval = Duration.ofSeconds(5);
			/** Specifies the maximum waiting time for the provisioning operation to finish. */
			private Duration pollingTimeout = Duration.ofMinutes(10);
		}

		@lombok.Getter
		@lombok.Setter
		public static class Mock extends Enabled {
			/** Path to the directory to start looking for SQLite database files. */
			private String sqliteDirectory;

			public Mock() {
				super(true);
			}
		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Persistence {

		/**
		 * Properties for persistence services.
		 * The key can be chosen arbitrarily and is used as the persistence service name, if the `name` property is not explicitly defined.
		 * In addition it can be leveraged to split configuration across multiple profiles.
		 */
		private Map<String, PersistenceServiceConfig> services = new HashMap<>();

		public Map<String, PersistenceServiceConfig> getServices() {
			services.forEach((k, v) -> {
				if(v.getName() == null || v.getName().trim().isEmpty()) {
					v.setName(k);
				}
			});
			return services;
		}

		@lombok.Getter
		@lombok.Setter
		public static class PersistenceServiceConfig extends Enabled {

			/** The name of the persistence service. */
			private String name;
			/** The name of the service binding used for this persistence service. If not set, the name is used. */
			private String binding;
			/** The name of the existing data source. If not set, the auto-configured data source of the binding is used. */
			private String dataSource;
			/** The name of the existing transaction manager. If not set, the transaction manager is automatically created. */
			private String transactionManager;

			public PersistenceServiceConfig() {
				// keep default c'tor for deserializion from property files
				super(true);
			}

			public PersistenceServiceConfig(String name) {
				this();
				this.name = name;
			}

		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Application {

		/**
		 * Properties for application services.
		 * The key can be chosen arbitrarily and is used as the application service name, if the `name` property is not explicitly defined.
		 * In addition it can be leveraged to split configuration across multiple profiles.
		 */
		private Map<String, ApplicationServiceConfig> services = new HashMap<>();

		public Map<String, ApplicationServiceConfig> getServices() {
			services.forEach((k, v) -> {
				if(v.getName() == null || v.getName().trim().isEmpty()) {
					v.setName(k);
				}
			});
			return services;
		}

		public ApplicationServiceConfig getService(String name) {
			return getServices().values().stream()
					.filter(s -> Objects.equals(s.getName(), name))
					.findFirst()
					.orElse(new ApplicationServiceConfig(name));
		}

		public List<ApplicationServiceConfig> getServicesByModel(String model) {
			return getServices().values().stream()
					.filter(s -> (Objects.equals(s.getModel(), model) || (s.getModel() == null && Objects.equals(s.getName(), model))))
					.collect(Collectors.toList());
		}

		@lombok.Getter
		@lombok.Setter
		public static class ApplicationServiceConfig {

			/** The name of the application service. */
			private String name;
			/**
			 * The qualified name of the CDS service, which is the model definition of this application service.
			 * It defaults to the name of the application service itself.
			 */
			private String model;
			/** Properties to configure how this service is served by protocol adapters. */
			private Serve serve = new Serve();

			public ApplicationServiceConfig() {
				// keep default c'tor for deserializion from property files
			}

			public ApplicationServiceConfig(String name) {
				this.name = name;
			}

			@lombok.Getter
			@lombok.Setter
			public static class Serve {

				/** Determines, if the service is ignored by protocol adapters. */
				private boolean ignore;
				/**
				 * The path this service should be served at by protocol adapters.
				 * The path is appended to the protocol adapter's base path.
				 * If a service is served by multiple protocol adapters, each adapter serves the service under this path.
				 */
				private String path;
				/**
				 * The list of protocols adapters this service should be served by.
				 * By default the service is served by all available protocol adapters.
				 * Possible values are: `odata-v4`, `odata-v2`.
				 */
				private List<String> protocols = new ArrayList<>();
				/**
				 * Properties to control more fine-grained under which endpoints this service is served.
				 * These properties override the more general properties `paths` and `protocols`.
				 */
				private List<Endpoint> endpoints = new ArrayList<>();

				@lombok.Getter
				@lombok.Setter
				public static class Endpoint {

					/**
					 * The path, this endpoint should be served at by the protocol adapter.
					 * The path is appended to the protocol adapter's base path.
					 */
					private String path;
					/**
					 * The protocol adapter that serves this endpoint.
					 * Possible values are: `odata-v4`, `odata-v2`.
					 */
					private String protocol;

				}

			}
		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Remote {

		/**
		 * Properties for remote services.
		 * The key can be chosen arbitrarily and is used as the application service name, if the `name` property is not explicitly defined.
		 * In addition it can be leveraged to split configuration across multiple profiles.
		 */
		private Map<String, RemoteServiceConfig> services = new HashMap<>();

		public Map<String, RemoteServiceConfig> getServices() {
			services.forEach((k, v) -> {
				if(v.getName() == null || v.getName().trim().isEmpty()) {
					v.setName(k);
				}
			});
			return services;
		}

		public RemoteServiceConfig getService(String name) {
			return getServices().values().stream()
					.filter(s -> Objects.equals(s.getName(), name))
					.findFirst()
					.orElse(new RemoteServiceConfig(name));
		}

		public List<RemoteServiceConfig> getServicesByModel(String model) {
			return getServices().values().stream()
					.filter(s -> (Objects.equals(s.getModel(), model) || (s.getModel() == null && Objects.equals(s.getName(), model))))
					.collect(Collectors.toList());
		}

		@lombok.Getter
		@lombok.Setter
		public static class RemoteServiceConfig {

			public static final String ODATA_V2_TYPE = "odata-v2";
			public static final String ODATA_V4_TYPE = "odata-v4";
			public static final String HCQL_TYPE = "hcql";

			/** The name of the remote service. */
			private String name;
			/**
			 * The qualified name of the CDS service, which is the model definition of this remote service.
			 * It defaults to the name of the remote service itself.
			 */
			private String model;
			/**
			 * The protocol type of the destination.
			 * Possible values are: `odata-v4`, `odata-v2` and `hcql`.
			 */
			private String type = ODATA_V4_TYPE;
			/** Properties to configure http requests for this remote service. */
			private Http http = new Http();
			/** Properties to configure a remote destination for this remote service. */
			private Destination destination = new Destination();
			/** Properties to configure a service binding for this remote service. **/
			private Binding binding = new Binding();

			public RemoteServiceConfig() {
				// keep default c'tor for deserializion from property files
			}

			public RemoteServiceConfig(String name) {
				this.name = name;
			}

			@lombok.Getter
			@lombok.Setter
			public static class Destination {

				/** The name of the destination in the destination service or SAP Cloud SDK destination accessor. */
				private String name;
				/**
				 * A map of generic destination properties supported by SAP Cloud SDK.
				 * These properties are used to dynamically build a destination.
				 */
				private Map<String, Object> properties = new HashMap<>();
				/**
				 * The retrieval strategy used, when loading destinations from SAP BTP Destination Service.
				 * See https://sap.github.io/cloud-sdk/docs/java/features/connectivity/sdk-connectivity-destination-service#retrieval-strategy-options for possible values.
				 */
				private String retrievalStrategy;
				/**
				 * The token exchange strategy used, when loading destinations from SAP BTP Destination Service.
				 * See https://sap.github.io/cloud-sdk/docs/java/features/connectivity/sdk-connectivity-destination-service#token-exchange-options for possible values.
				 */
				private String tokenExchangeStrategy;

			}

			@lombok.Getter
			@lombok.Setter
			public static class Binding {

				/** The name of the service binding which provides the credentials. */
				private String name;
				/**
				 * Determines the user and tenant context that is used.
				 * Possible values are: `currentUser`, `systemUser` and `systemUserProvider`
				 */
				private String onBehalfOf = "currentUser";
				/**
				 * A map of additional options that are used to enrich the used service binding. The supported options depend on the type of service binding that is used.
				 * Possible options are: `url` for service bindings of type `xsuaa`
				 */
				private Map<String, String> options = new HashMap<>();

			}

			@lombok.Getter
			@lombok.Setter
			public static class Http {

				/** A suffix for the remote service, that is appended to the remote service's URL. */
				private String suffix;
				/**
				 * The name of the modelled service, that is appended to the remote service's URL (after the suffix).
				 * It defaults to the qualified name of the model definition.
				 */
				private String service;
				/** A map of headers and their values, to be added to every outgoing request. */
				private Map<String, String> headers = new HashMap<>();
				/** A map of query parameters and their values, to be added to every outgoing request. */
				private Map<String, String> queries = new HashMap<>();
				/** Properties to configure CSRF token requests for this service. */
				private Enabled csrf = new Enabled(false);

			}
		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class Locales {

		/** Properties to configure how locales should be normalized. */
		private Normalization normalization = new Normalization();

		@lombok.Getter
		@lombok.Setter
		public static class Normalization {

			/** Determines, if the non-normalization include list, as described in the documentation, is applied. */
			private boolean defaults = true;
			/** The list of additional locales to add to the include list of non-normalized locales. */
			private List<String> includeList = new ArrayList<>();

		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Errors {

		/**
		 * Properties to configure how error messages from the framework are treated.
		 * If turned off, only framework error messages, that are explicitly localized are returned.
		 * Other errors are mapped to their plain HTTP error code representation.
		 */
		private Enabled stackMessages = new Enabled(true);
		/** Determines, if error messages are automatically extended with additional debug information (only for development). */
		private boolean extended = false;
		/** Determines, if validation error messages are collected and exceptions are thrown at the end of the Before event handler phase. */
		private boolean combined = true;

	}

	/** Properties for drafts. */
	@lombok.Getter
	@lombok.Setter
	public static class Drafts {

		/** The maximum amount of time, since the last change, an entity instance is locked by the user who is editing its draft version. */
		private Duration cancellationTimeout = Duration.ofMinutes(15);
		/** The maximum amount of time a draft is kept, before it is garbage collected. */
		private Duration deletionTimeout = Duration.ofDays(30);
		/**
		 * Determines the persistence mode for draft-enabled entities. In case of `joint` persistence
		 * queries with JOINs between databases tables/views of active instances and databases tables of draft instances
		 * can occur. If set to `split` such queries are avoided.
		 * Possible values are: `joint`, `split`.
		 */
		private String persistence = "joint";
		/** Properties to configure the automatic draft garbage collection. */
		private GC gc = new GC();

		@lombok.Getter
		@lombok.Setter
		public static class GC extends Enabled {

			/** The interval, in which the automatic draft garbage collection is triggered. */
			private Duration interval = Duration.ofHours(6);
			/** The max duration the draft GC timer task pauses in between processing GCs of different tenants. */
			private Duration maxPause = Duration.ofMinutes(5);

			public GC() {
				super(true);
			}

		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Query {

		/** Properties for server-driven paging. */
		private Limit limit = new Limit();
		/** Properties for the implicit-sorting feature. */
		private Enabled implicitSorting = new Enabled(true);

		@DocumentedProperty(false)
		private Validation validation = new Validation();

		public static class Limit {
			/**
			 * The default page size for server-driven paging.
			 * Setting this property to 0 or -1 disables the default page size.
			 */
			private int _default = 0; // can't use lombok on this, as default is an invalid attribute name
			/**
			 * The maximum page size for server-driven paging.
			 * Setting this property to 0 or -1 disables the maximum page size.
			 */
			@lombok.Getter
			@lombok.Setter
			private int max = 1000;

			public int getDefault() {
				return _default;
			}

			public void setDefault(int _default) {
				this._default = _default;
			}

			/**
			 * Properties for reliable server-driven paging, based on last row of ordered result.
			 */
			@lombok.Getter
			@lombok.Setter
			private Enabled reliablePaging = new Enabled(false);
		}

		@lombok.Getter
		@lombok.Setter
		public static class Validation {

			/** Disables validation of function and actions parameters. */
			private Enabled parameters = new Enabled(true);

		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class Sql {

		/**
		 * The JDBC batch size used for batch and bulk operations.
		 */
		private int maxBatchSize = 1000;

		@DocumentedProperty(false)
		private String collate = "strings";

		/**
		 * Configuration properties for an SQL generation on SAP HANA.
		 */
		private Hana hana = new Hana();

		/**
		 * Configuration properties for an SQL generation related to inline count.
		 */
		private InlineCount inlineCount = new InlineCount();

		@lombok.Getter
		@lombok.Setter
		public static class InlineCount {
			/**
			 * Sets the inline-count mode.
			 * Possible values are: `auto`, `window-function` and `query`
			 */
			private String mode = "auto";
		}

		/**
		 * Configuration properties for an SQL generation related to search.
		 */
		@DocumentedProperty(false)
		private Search search = new Search();

		@lombok.Getter
		@lombok.Setter
		public static class Search {

			/**
			 * Sets the search mode.
			 * Possible values are: `generic`, `localized-view`, `localized-association`
			 */
			private String mode;

		}

		@lombok.Getter
		@lombok.Setter
		public static class Hana {

			/**
			 * If set to `true`, this property completely disables locale specific handling on SAP HANA.
			 */
			private boolean ignoreLocale = false;

			/**
			 * Sets the optimization mode for SAP HANA. The mode `hex` generates SQL that is
			 * optimized exclusively for the HANA Cloud HEX engine. The mode `legacy`
			 * generates SQL, which is not optimized for the HEX engine and can also run on
			 * SAP HANA 2.x (HaaS).
			 */
			private String optimizationMode = "hex";

			/**
			 * Properties for the HANA docstore support
			 */
			@DocumentedProperty(false)
			private Enabled docstore = new Enabled(false);

			/**
			 * Properties for search on SAP HANA Cloud.
			 */
			private Search search = new Search();

			@lombok.Getter
			@lombok.Setter
			public static class Search {
				/**
				 * If set to `true`, fuzzy search on SAP HANA Cloud is activated.
				 * Requires `cds.sql.hana.optimizationMode: hex`
				 */
				private boolean fuzzy = false;

				/**
				 * The least level of fuzziness, searchable elements need to have to be considered in a fuzzy search.
				 * The value can be in the inclusive range of 0 to 1, where 1 means no fuzziness (exact search).
				 */
				private BigDecimal fuzzinessThreshold = new BigDecimal("0.8");
			}
		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class AuditLog {

		/** Configuration of the AuditLog V2 feature. */
		private V2 v2 = new V2();

		/** Configuration of the Personal Data handler. */
		private PersonalData personalData = new PersonalData();

		/** Properties to configure AuditLog usage of Outbox.*/
		private Outbox outbox = new Outbox();

		/** Configuration for standard events */
		@DocumentedProperty(false)
		private StandardEvents standardEvents = new StandardEvents();

		/** Properties to configure the HTTP connection pool for AuditLog usage. */
		private ConnectionPool connectionPool = new ConnectionPool();
		@lombok.Getter
		@lombok.Setter
		public static class PersonalData extends Enabled {

			/** If set to {@code true} and the data subject is missing, a `ServiceException` is thrown. */
			private boolean throwOnMissingDataSubject = true;

			/** If set to {@code true}, audit logging of access to sensitive personal data is enabled. */
			private Enabled logRead = new Enabled(true);

			@DocumentedProperty(false)
			private Enabled logInsert = new Enabled(true);
			@DocumentedProperty(false)
			private Enabled logDelete = new Enabled(true);
			@DocumentedProperty(false)
			private Enabled logUpdate = new Enabled(true);

			public PersonalData() {
				super(true);
			}
		}

		@lombok.Getter
		@lombok.Setter
		public static class Outbox extends Enabled {

			/** The name of the outbox service to use. If not available or set, the in-memory outbox is used as fallback. */
			private String name = OutboxService.PERSISTENT_UNORDERED_NAME;

			public Outbox() {
				super(true);
			}
		}

		@lombok.Getter
		@lombok.Setter
		public static class V2 extends Enabled {
			public V2() {
				super(true);
			}

			/** Use logon name as user for audit log messages. */
			@DocumentedProperty(false)
			private boolean useLogonName = false;
		}

		/** Enable or disable logging of standard events */
		@lombok.Getter
		@lombok.Setter
		public static class StandardEvents extends Enabled {
			public StandardEvents() {
				super(false);
			}

			/** Switch on/off logging of certain standard events */
			private Enabled tenantLifecycleEvents = new Enabled(false);
			private Enabled unauthorizedRequestEvent = new Enabled(true);
		}
	}
	@lombok.Getter
	@lombok.Setter
	public static class Outbox {

		/** Properties for the in-memory Outbox. */
		private InMemory inMemory = new InMemory();

		/** Properties for the persistent Outbox. */
		private Enabled persistent = new Enabled(true);

		@DocumentedProperty(false)
		private Duration tenantsInfoRefreshInterval = Duration.ofMinutes(5);
		@DocumentedProperty(false)
		private Enabled providerTenant = new Enabled(true);

		/**
		 * Properties for Outbox services.
		 * The key can be chosen arbitrarily and is used as the outbox service name, if the `name` property is not explicitly defined.
		 * In addition it can be leveraged to split configuration across multiple profiles.
		 */
		private Map<String, OutboxServiceConfig> services = new HashMap<>();

		public Map<String, OutboxServiceConfig> getServices() {
			services.forEach((k, v) -> {
				if (v.getName() == null || v.getName().trim().isEmpty()) {
					v.setName(k);
				}
			});

			return services;
		}

		public OutboxServiceConfig getService(String name) {
			return getServices().values().stream()
					.filter(s -> Objects.equals(s.getName(), name))
					.findFirst()
					.orElse(new OutboxServiceConfig(name));
		}

		@lombok.Getter
		@lombok.Setter
		public static class InMemory extends Enabled {

			public InMemory() {
				super(true);
			}

			/**
			 * Specifies whether the outbox emits the event during the current `ChangeSetContext` or afterwards.
			 * In case the `ChangeSetContext` wraps a DB transaction, it needs to be considered that
			 * emitted messages during the transaction can't be rolled back on cancelled transaction.
			 * Messages being emitted after the transaction might get lost, after transaction has been committed successful.
			 */
			private boolean emitDuringChangeSetContext = false;
		}

		@lombok.Getter
		@lombok.Setter
		public static class OutboxServiceConfig extends Enabled	{

			public OutboxServiceConfig(String name) {
				super(true);
				this.name = name;
			}

			public OutboxServiceConfig() {
				// keep default c'tor for deserializion from property files
				super(true);
			}

			/**
			 * The name of the Outbox service.
			 */
			private String name;

			/**
			 * Specifies the maximum number of attempts to emit a message stored in the Outbox.
			 * Messages that have reached the maximum number of attempts are ignored by the Outbox and need to be handled by the application.
			 */
			private int maxAttempts = 10;

			@DocumentedProperty(false)
			private Duration emitTimeout = Duration.ofSeconds(10);
			@DocumentedProperty(false)
			private int chunkSize = 10;
			@DocumentedProperty(false)
			private Duration maxPause = Duration.ofMinutes(10);
			@DocumentedProperty(false)
			private Enabled triggerSchedule = new Enabled(true);
			@DocumentedProperty(false)
			private boolean startCollector = true;

			/** Controls storing the error message of the last error in the Outbox. */
			private Enabled storeLastError = new Enabled(true);
			/**
			 * Controls how messages are published by the outbox. If set to `true`, the messages are sequentially published exactly in the order they were stored in the Outbox.
			 * If set to `false`, the messages are allowed to be published in random order and in parallel.
			 */
			private boolean ordered = true;
			/** Controls whether the outbox records telemetry information for observability purposes. */
			private boolean observable = true;
		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class CloudSdk {

		/** Properties for the RequestContext integration with Cloud SDK's ExecutorService. */
		private Enabled executorDecorator = new Enabled(true);

	}

	/*
	 * COMMON CONFIG POJOS
	 */
	@lombok.Setter
	public static class Enabled {

		/** Determines, if it is enabled. */
		private Boolean enabled; // can't use lombok on this, as it would generate getEnabled() instead of isEnabled()

		public Enabled(Boolean enabled) {
			this.enabled = enabled;
		}

		public Boolean isEnabled() {
			return enabled;
		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class ConnectionPool {
		/**
		 * The max amount of connections in the pool.
		 */
		private Integer maxConnections = 200;

		/**
		 * The max amount of connections from the pool per route
		 */
		private Integer maxConnectionsPerRoute = 20;

		/**
		 *  The threshold for connect timeout, socket timeout and connection request timeout.
		 */
		private Duration timeout = Duration.ofMillis(60000);

		/**
		 * Properties to control how the http connection pools are maintained.
		 * This allows to configure, whether all tenants will share a common http connection pool instead of having a dedicated http connection pool for each tenant.
		 */
		private Enabled combinePools = new Enabled(true);

		public ConnectionPool() {
		}

		public ConnectionPool(Duration timeout, Integer maxConnectionsPerRoute, Integer maxConnections) {
			this.timeout = timeout;
			this.maxConnectionsPerRoute = maxConnectionsPerRoute;
			this.maxConnections = maxConnections;
		}
	}
	@lombok.Getter
	@lombok.Setter
	public static class Servlet extends Enabled {

		/** The base-path of the adapter endpoint. */
		private String path;

		public Servlet(String path, boolean enabled) {
			super(enabled);
			this.path = path;
		}

	}

}
