/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.messaging;

import com.sap.cds.services.Service;
import java.util.Map;

/** Messaging service. */
public interface MessagingService extends Service {

  /** The name of the composite {@link MessagingService} */
  public static final String COMPOSITE_NAME = "MessagingService$Composite";

  /**
   * The messaging error event, that allows to handle acknowledgement of a message on the messaging
   * channel
   */
  public static final String EVENT_MESSAGING_ERROR = "MESSAGING_ERROR";

  /**
   * If the service is not configured with the structured flag, the message is sent to the specified
   * topic of this message broker as is.
   *
   * <p>If the service is configured with the structured flag (default), the message String is
   * converted into a Map following the rule: {message: message} The Map is then interpreted as data
   * map and passed to {@link #emit(String, Map)}. Usually this results in a final message like:
   * {data: {message: message}}
   *
   * @param topic the topic
   * @param message the message to be sent
   * @deprecated Use {@link #emit(String, Map)} or {@link #emit(String, Map, Map)} instead.
   */
  @Deprecated(forRemoval = true, since = "3.0.0")
  public void emit(String topic, String message);

  /**
   * If the service is not configured with the structured flag, the message Map is serialized to
   * JSON and sent to the specified topic of this message broker as is.
   *
   * <p>If the service is configured with the structured flag (default), the message Map is
   * interpreted as data map and passed to {@link #emit(String, Map, Map)}. Usually this results in
   * a final message like: {data: message}
   *
   * @param topic the topic
   * @param message the Map to be serialized to JSON and then sent
   */
  public void emit(String topic, Map<String, Object> message);

  /**
   * Takes a (cloudevents) message, separated into data and headers and sends it to the specified
   * topic of this message broker. This method produces the same final message, regardless of the
   * structured flag.
   *
   * <p>Usually data and headers are combined into a final JSON string message following the rule:
   * {...headers, data: data}. Brokers (e.g. Kafka) that do natively support headers are able to
   * separate headers from data, when using this method.
   *
   * @param topic the topic
   * @param data the data Map
   * @param headers the headers Map
   */
  public void emit(String topic, Map<String, Object> data, Map<String, Object> headers);
}
