/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.runtime;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.authentication.AuthenticationInfo;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.environment.CdsEnvironment;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import java.util.Locale;

/**
 * Interface for Protocol Adapters to get access to {@link Service} instances and {@link CdsModel}.
 * It can be used to define Request and ChangeSet boundaries.
 */
public interface CdsRuntime {

  /**
   * @return The {@link CdsEnvironment} built from the environment providers
   */
  CdsEnvironment getEnvironment();

  /**
   * @return The {@link ServiceCatalog} containing all registered {@link Service} instances
   */
  ServiceCatalog getServiceCatalog();

  /**
   * @return The common {@link CdsModel} as read from CSN file
   */
  CdsModel getCdsModel();

  /**
   * @param tenant the tenant
   * @return the CDS model for a specific tenant or the default model if {@code tenant} is {@code
   *     null}
   */
  default CdsModel getCdsModel(String tenant) {
    return getCdsModel(UserInfo.create().setTenant(tenant), FeatureTogglesInfo.create());
  }

  /**
   * @param userInfo the {@link UserInfo}
   * @param featureTogglesInfo the {@link FeatureTogglesInfo}
   * @return The CDS model for a specific tenant and {@link FeatureTogglesInfo} or the default model
   *     if the tenant is {@code null}
   */
  CdsModel getCdsModel(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo);

  /**
   * @return The {@link ParameterInfo} retrieved from the registered {@link ParameterInfoProvider}
   *     or an empty {@link ParameterInfo}.
   */
  ParameterInfo getProvidedParameterInfo();

  /**
   * @return The {@link UserInfo} retrieved from the registered {@link UserInfoProvider} or an
   *     anonymous {@link UserInfo}.
   */
  UserInfo getProvidedUserInfo();

  /**
   * @return The {@link AuthenticationInfo} retrieved from the registered {@link
   *     AuthenticationInfoProvider} or {@code null} if no authentication was provided.
   */
  AuthenticationInfo getProvidedAuthenticationInfo();

  /**
   * @param userInfo the {@link UserInfo} object
   * @param parameterInfo the {@link ParameterInfo}
   * @return The {@link FeatureTogglesInfo} retrieved from the registered {@link
   *     FeatureTogglesInfoProvider}
   */
  FeatureTogglesInfo getFeatureTogglesInfo(UserInfo userInfo, ParameterInfo parameterInfo);

  /**
   * Returns the localized message for the given code, arguments and locale if existing, {@code
   * null} otherwise. The method does not throw and returns {@code code} in case of any errors.
   *
   * @param code The code which is used as key for the format string in the defined bundles.
   * @param args The argument objects for the format string
   * @param locale The locale to be used. If no locale is set the default locale is taken.
   * @return The localized message or {@code null} if the code is not resolvable.
   */
  String getLocalizedMessage(String code, Object[] args, Locale locale);

  /**
   * Returns the {@link CdsProvider} of type {@code T}
   *
   * @param <T> the specific {@link CdsProvider} type
   * @param clazz the class of the type of {@link CdsProvider}
   * @return the {@link CdsProvider} of type {@code T}
   */
  <T extends CdsProvider<T>> T getProvider(Class<T> clazz);

  /**
   * Starts a new {@link RequestContext} which defines the context of parameters (reflecting the
   * user, request parameters etc.)
   *
   * @return the {@link RequestContextRunner} used to open the context.
   */
  RequestContextRunner requestContext();

  /**
   * Starts a new {@link ChangeSetContext} which defines the transactional boundaries
   *
   * @return the {@link ChangeSetContextRunner} used to open the context.
   */
  ChangeSetContextRunner changeSetContext();
}
