/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.request;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Interface to modify the values of a {@link UserInfo} used in a {@link com.sap.cds.services.request.RequestContext}..
 */
public interface ModifiableUserInfo extends UserInfo {

	/**
	 * Sets the ID of the user.
	 *
	 * @param id	The ID to set.
	 * @return	The same {@code User}.
	 */
	ModifiableUserInfo setId(String id);

	/**
	 * Sets the name of the user which is used for $user to uniquely identify.
	 *
	 * @param name	The name to set.
	 * @return	The same {@code User}.
	 */
	ModifiableUserInfo setName(String name);

	/**
	 * Sets the tenant of the user.
	 *
	 * @param tenant	The tenant to set.
	 * @return	The same {@code User}.
	 */
	ModifiableUserInfo setTenant(String tenant);

	/**
	 * Adds a role to the user.
	 *
	 * @param role	The role to add.
	 * @return	The same {@code User}.
	 */
	ModifiableUserInfo addRole(String role);

	/**
	 * Removes a role from the user.
	 *
	 * @param role	The role to remove.
	 * @return	The same {@code User}.
	 */
	ModifiableUserInfo removeRole(String role);

	/**
	 * Sets the roles of the user. {@code null} is allowed.
	 *
	 * @param roles	The roles to set.
	 * @return	The same {@code User}.
	 */
	ModifiableUserInfo setRoles(Set<String> roles);

	/**
	 * Sets the  value list of an attribute of the user. {@code null} is allowed.
	 *
	 * @param attribute	The name of the attribute.
	 * @param values	The values of the attribute.
	 * @return	The same {@code User}.
	 */
	ModifiableUserInfo setAttributeValues(String attribute, List<String> values);

	/**
	 * Sets an attribute value of the user. {@code null} is allowed.
	 *
	 * @param attributes	The attributes to set.
	 * @return	The same {@code User}.
	 */
	ModifiableUserInfo setAttributes(Map<String, List<String>> attributes);

	/**
	 * Sets the additional attributes.
	 * @param additionalAttributes	The attributes
	 * @return	The same {@code User}.
	 */
	ModifiableUserInfo setAdditionalAttributes(Map<String, Object> additionalAttributes);

	/**
	 * Sets additional attributes.
	 *
	 * @param name	The name of the attribute
	 * @param value	The value of the attribute
	 * @return	The same {@code User}.
	 */
	ModifiableUserInfo setAdditionalAttribute(String name, Object value);

	/**
	 * Specifies if the current user is an authenticated (i.e. business) user.
	 * @param val The new value
	 * @return The same {@code User}.
	 */
	ModifiableUserInfo setIsAuthenticated(boolean val);

	/**
	 * Specifies if the current user is a system user.
	 * @param val The new value
	 * @return The same {@code User}.
	 */
	ModifiableUserInfo setIsSystemUser(boolean val);

	/**
	 * Specifies if the current user is an internal user.
	 * @param val The new value
	 * @return The same {@code User}.
	 */
	ModifiableUserInfo setIsInternalUser(boolean val);
}
