/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.messages;

import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Stream;

import com.sap.cds.services.ServiceException;
import com.sap.cds.services.request.RequestContext;

/**
 * API to create {@code Message}s with different severity within the {@link RequestContext}.
 * {@code Message}s provide additional information which may be added to the request's response.
 *
 * @see RequestContext#getMessages()
 */
public interface Messages {

	/**
	 * Adds a new {@code Message} with severity {@link Message.Severity#INFO} to the current request with the given message text or format and format parameters.
	 * In case a key to a message format is passed, the request's locale is taken for localization.
	 *
	 * @param messageOrKey	The message format or a key to a message format in the resource bundles.
	 * @param params	The parameters to fill the message format.
	 * @return	The resulting {@link Message} object which can be adapted accordingly
	 */
	Message info(String messageOrKey, Object...params);

	/**
	 * Adds a new {@code Message} with severity {@link Message.Severity#SUCCESS} to the current request with the given message text or format and format parameters.
	 * In case a key to a message format is passed, the request's locale is taken for localization.
	 *
	 * @param messageOrKey	The message format or a key to a message format in the resource bundles.
	 * @param params	The parameters to fill the message format.
	 * @return	The resulting {@link Message} object which can be adapted accordingly
	 */
	Message success(String messageOrKey, Object...params);

	/**
	 * Adds a new {@code Message} with severity {@link Message.Severity#WARNING} to the current request with the given message text or format and format parameters.
	 * In case a key to a message format is passed, the request's locale is taken for localization.
	 *
	 * @param messageOrKey	The message format or a key to a message format in the resource bundles.
	 * @param params	The parameters to fill the message format.
	 * @return	The resulting {@link Message} object which can be adapted accordingly
	 */
	Message warn(String messageOrKey, Object...params);

	/**
	 * Adds a new {@code Message} with severity {@link Message.Severity#ERROR} to the current request with the given message text or format and format parameters.
	 * In case a key to a message format is passed, the request's locale is taken for localization.
	 *
	 * @param messageOrKey	The message format or a key to a message format in the resource bundles.
	 * @param params	The parameters to fill the message format.
	 * @return	The resulting {@link Message} object which can be adapted accordingly
	 */
	Message error(String messageOrKey, Object...params);

	/**
	 * Returns a {@link Stream} of the added messages in the current request context
	 *
	 * @return	The {@link Stream} of the added messages
	 */
	Stream<Message> stream();

	/**
	 * Removes all messages matching the given filter.
	 *
	 * @param filter the filter matching messages that should be removed
	 * @return The list of removed messages
	 */
	List<Message> removeIf(Predicate<Message> filter);

	/**
	 * Throws a service exception, if error messages have been added to the current request.
	 *
	 * @throws ServiceException if there was at least one error message added to the current request
	 */
	void throwIfError() throws ServiceException;

}
