/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services;

import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.function.Function;

import org.slf4j.helpers.MessageFormatter;

import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.handler.Handler;
import com.sap.cds.services.messages.MessageLookup;
import com.sap.cds.services.messages.MessageTarget;

/**
 * {@link ServiceException} is the central unchecked exception thrown by the framework and {@link Handler} when an error occurs during event processing via {@link Service#emit(EventContext)}
 * It extends {@link RuntimeException} with an {@link ErrorStatus}, which indicates an internal error code and a mapping to an HTTP status code.
 */
public class ServiceException extends RuntimeException {

	private static final long serialVersionUID = 1L;

	protected final ErrorStatus errorStatus;
	private final MessageLookup messageLookup;
	private String longTextUrl;
	private MessageTarget messageTarget;
	private List<MessageTarget> additionalTargets = List.of();
	private boolean isTransition = true;

	protected final Object[] args;

	protected static ServiceExceptionUtils Utils = CoreFactory.INSTANCE.createServiceExceptionUtils();

	/**
	 * Creates a new {@link ServiceException}
	 * <br>
	 * The {@link ErrorStatus} of the first {@link ServiceException} found within the cause chain is used.
	 * If no {@link ErrorStatus} is found, it defaults to {@link ErrorStatuses#SERVER_ERROR}.
	 *
	 * @param e The causing {@link Throwable}
	 */
	public ServiceException(Throwable e) {
		this(null, e.getMessage(), new Object[] { e });
	}

	/**
	 * Creates a new {@link ServiceException}. The last argument might be the causing {@link Throwable} and not a formatting argument.
	 * <br>
	 * The {@link ErrorStatus} of the first {@link ServiceException} found within the cause chain is used.
	 * If no {@link ErrorStatus} is found, it defaults to {@link ErrorStatuses#SERVER_ERROR}.
	 *
	 * @param messageOrKey The formatting message, based on SLF4J's {@link MessageFormatter}, or a key to a message format in the resource bundles.
	 * @param args The argument objects for the format string. The last argument might be the causing {@link Throwable}.
	 */
	public ServiceException(String messageOrKey, Object... args) {
		this(null, messageOrKey, args);
	}

	/**
	 * Creates a new {@link ServiceException}
	 * @param errorStatus 	The {@link ErrorStatus}, if <code>null</code> the {@link ErrorStatus} of the first {@link ServiceException} found within the cause chain is used.
	 * 						If no {@link ErrorStatus} is found, it defaults to {@link ErrorStatuses#SERVER_ERROR}
	 * @param messageOrKey 	The formatting message, based on SLF4J's {@link MessageFormatter}, or a key to a message format in the resource bundles.
	 * @param args 			The argument objects for the format string. The last argument might be the causing {@link Throwable}.
	 */
	public ServiceException(ErrorStatus errorStatus, String messageOrKey, Object... args) {
		super(messageOrKey);

		Throwable throwableCandidate = MessageFormatter.getThrowableCandidate(args);
		if (throwableCandidate != null) {
			super.initCause(throwableCandidate);
			this.args = MessageFormatter.trimmedCopy(args);
		} else {
			this.args = args; // NOSONAR
		}
		this.errorStatus = errorStatus;
		this.messageLookup = MessageLookup.create(messageOrKey, this.args);
	}

	@Override
	public String getMessage() {
		return Utils.getMessage(getPlainMessage(), args);
	}

	@Override
	public String getLocalizedMessage() {
		return getLocalizedMessage(null);
	}

	public String getLocalizedMessage(Locale locale) {
		return Utils.getLocalizedMessage(getPlainMessage(), args, locale, true);
	}

	/**
	 * @return The original message of this exception
	 */
	public String getPlainMessage() {
		return super.getMessage();
	}

	/**
	 * Returns the stack of {@link EventContext} which corresponds to the events processed along the exception stack trace.
	 * The lists starts with the {@link EventContext} from which the exception was triggered.
	 * If there are no contexts, the method returns an empty list
	 *
	 * @return the stack of {@link EventContext} which corresponds to the events processed along the exception stack trace.
	 */
	public List<EventContext> getEventContexts() {
		List<EventContext> contexts = getNearest(ServiceException::collectEventContexts);
		return contexts == null ? Collections.emptyList() : Collections.unmodifiableList(contexts);
	}

	protected List<EventContext> collectEventContexts() {
		return null; // getNearest works with nulls
	}

	/**
	 * Iterates over all {@link ServiceException} in the chain and returns the nearest available value of an attribute
	 * @param <T> the attribute type
	 * @param provider the provider for the attribute
	 * @return the attribute value
	 */
	private <T> T getNearest(Function<ServiceException, T> provider) {
		T value = provider.apply(this);
		Throwable cause = getCause();
		while (value == null && cause != null) {
			if (cause instanceof ServiceException exception) {
				value = provider.apply(exception);
			}
			cause = cause.getCause();
		}
		return value;
	}

	/**
	 * @return the {@link ErrorStatus}
	 */
	public ErrorStatus getErrorStatus() {
		ErrorStatus theErrorStatus = getNearest(e -> e.errorStatus);
		return theErrorStatus == null ? ErrorStatuses.SERVER_ERROR : theErrorStatus;
	}

	/**
	 * @return the {@link MessageLookup}
	 */
	public MessageLookup getMessageLookup() {
		return messageLookup;
	}

	/**
	 * @return the long text url
	 */
	public String getLongTextUrl() {
		return getNearest(e -> e.longTextUrl);
	}

	/**
	 * @return the {@link MessageTarget}
	 */
	public MessageTarget getMessageTarget() {
		return getNearest(e -> e.messageTarget);
	}

	/**
	 * @return the list of additional {@link MessageTarget} instances
	 */
	public List<MessageTarget> getAdditionalTargets() {
		return getNearest(e -> e.additionalTargets);
	}

	/**
	 * Returns {@code true}, if the transition indicator is set for this exception.
	 *
	 * A transition message is a request-specific message, that is valid during the runtime of a request.
	 * It might not reoccur if the same request is processed again.
	 * A state message has a relation to an entity document.
	 * It is typically a validation (error) message.
	 *
	 * @return {@code true}, if the transition indicator is set for this exception
	 */
	public boolean isTransition() {
		return getNearest(e -> e.isTransition);
	}

	/**
	 * Adds the passed {@code longTextUrl} to the current exception.
	 *
	 * @param longTextUrl	The long text url to be set
	 * @return	The current {@link ServiceException}
	 */
	public ServiceException longTextUrl(String longTextUrl) {
		this.longTextUrl = longTextUrl;
		return this;
	}

	/**
	 * Sets the {@code transition} indicator of this exception.
	 *
	 * A transition message is a request-specific message, that is valid during the runtime of a request.
	 * It might not reoccur if the same request is processed again.
	 * A state message has a relation to an entity document.
	 * It is typically a validation (error) message.
	 *
	 * @param isTransition {@code true}, if the transition indicator should be set, {@code false}, if it should be unset
	 * @return	The current {@link ServiceException}
	 */
	public ServiceException transition(boolean isTransition) {
		this.isTransition = isTransition;
		return this;
	}

	/**
	 * Sets the provided {@link MessageTarget}
	 *
	 * @param target the {@link MessageTarget}
	 * @return The current {@link ServiceException}
	 */
	public ServiceException messageTarget(MessageTarget target) {
		this.messageTarget = target;
		return this;
	}

	/**
	 * Sets the provided string-based target. No further processing of the string is performed.
	 *
	 * @param target the string-based target
	 * @return The current {@link ServiceException}
	 */
	public ServiceException messageTarget(String target) {
		return messageTarget(MessageTarget.create(target));
	}

	/**
	 * Adds the passed path as target to the current {@link ServiceException}.
	 *
	 * The path is interpreted relative to the CQN statement, which was determined from the request.
	 * For CRUD events this CQN statement points to the targeted entity.
	 * For bound actions or functions this CQN statement points to the bound entity.
	 *
	 * Is equivalent to calling {@code messageTarget(MessageTarget.PARAMETER_CQN, path)}
	 *
	 * @param path the path to the target element or association
	 * @return The current {@link ServiceException}
	 */
	public ServiceException messageTarget(Function<StructuredType<?>, Object> path) {
		return messageTarget(MessageTarget.create(path));
	}

	/**
	 * Adds the passed target parameter and path as target to the current
	 * {@link ServiceException}.
	 *
	 * @param parameter target parameter serving as the entry point for the path resolution.
	 *                  Passing {@link MessageTarget#PARAMETER_CQN} indicates that the path
	 *                  should be interpreted relatively to the target entity of the request.
	 * 					Alternatively you can pass names of action or function parameters.
	 *
	 * @param path      the path to the target element or association
	 * @return The current {@link ServiceException}
	 */
	public ServiceException messageTarget(String parameter, Function<StructuredType<?>, Object> path) {
		return messageTarget(MessageTarget.create(parameter, path));
	}

	/**
	 * Adds the passed path as target to the current {@link ServiceException}.
	 * This method allows to build the path in a type-safe way, by passing the corresponding entity or structured type interface.
	 * <p>
	 * The path is interpreted relative to the CQN statement, which was determined from the request.
	 * For CRUD events this CQN statement points to the targeted entity.
	 * For bound actions or functions this CQN statement points to the bound entity.
	 * <p>
	 * Is equivalent to calling {@code messageTarget(MessageTarget.PARAMETER_CQN, type, path)}
	 *
	 * @param type the root type of the path. Either an entity or a structured type.
	 * @param path the path to the target element or association
	 * @param <E>  the type of the root
	 * @return The current {@link ServiceException}
	 */
	public <E extends StructuredType<E>> ServiceException messageTarget(Class<E> type, Function<E, Object> path) {
		return messageTarget(MessageTarget.create(type, path));
	}

	/**
	 * Adds the passed target parameter and path as target to the current
	 * {@link ServiceException}. This method allows to build the path in a type-safe way,
	 * by passing the corresponding entity or structured type interface.
	 *
	 * @param parameter target parameter serving as the entry point for the path resolution.
	 *                  Passing {@link MessageTarget#PARAMETER_CQN} indicates that the path
	 *                  should be interpreted relatively to the target entity of the request.
	 *                  Alternatively you can pass names of action or function parameters.
	 *
	 * @param type the root type of the path. Either an entity or a structured type.
	 * @param path the path to the target element or association
	 * @param <E>  the type of the root
	 * @return The current {@link ServiceException}
	 */
	public <E extends StructuredType<E>> ServiceException messageTarget(String parameter, Class<E> type, Function<E, Object> path) {
		return messageTarget(MessageTarget.create(parameter, type, path));
	}

	/**
	 * Adds the passed path and element as target to the current {@link ServiceException}.
	 * <p>
	 * The path is interpreted relative to the CQN statement, which was determined from the request.
	 * For CRUD events this CQN statement points to the targeted entity.
	 * For bound actions or functions this CQN statement points to the bound entity.
	 * <p>
	 * This method can be used with the {@link com.sap.cds.CdsDataProcessor}
	 * and its functional interfaces.
	 *
	 * @param path    the path
	 * @param element the target element or association
	 * @return The current {@link ServiceException}
	 */
	public ServiceException messageTarget(Path path, CdsElement element) {
		return messageTarget(MessageTarget.create(path, element));
	}

	/**
	 * Sets the provided list of {@link MessageTarget} instances as additional targets.
	 *
	 * @param additionalTargets the list of additional {@link MessageTarget} instances.
	 * @return The current {@link ServiceException}
	 */
	public ServiceException additionalTargets(List<MessageTarget> additionalTargets) {
		this.additionalTargets = additionalTargets;
		return this;
	}

	/**
	 * Sets the provided array of {@link MessageTarget} instances as additional targets.
	 *
	 * @param additionalTargets the array of additional {@link MessageTarget} instances.
	 * @return The current {@link ServiceException}
	 */
	public ServiceException additionalTargets(MessageTarget... additionalTargets) {
		return additionalTargets(List.of(additionalTargets));
	}

}
