/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.cds;

import java.util.Map;

import com.sap.cds.Result;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnUpsert;
import com.sap.cds.services.Service;

/**
 * The interface defining the consumption API of a {@link CqnService}.
 * Such a service is capable of executing CQN queries.
 */
public interface CqnService extends Service {

	/*
	 * Default events of the CQN Service
	 */
	public static final String EVENT_READ = "READ";
	public static final String EVENT_CREATE = "CREATE";
	public static final String EVENT_UPDATE = "UPDATE";
	public static final String EVENT_DELETE = "DELETE";
	public static final String EVENT_UPSERT = "UPSERT";

	/**
	 * Executes a {@link CqnSelect} statement with optional positional parameter values against the {@link CqnService}.
	 * @param select the {@link CqnSelect} to be executed
	 * @param paramValues the optional positional parameter values
	 * @return the {@link Result} of the query
	 */
	public Result run(CqnSelect select, Object... paramValues);

	/**
	 * Executes a {@link CqnSelect} statement with named parameter values against the {@link CqnService}.
	 * @param select the {@link CqnSelect} to be executed
	 * @param namedValues the named parameter values
	 * @return the {@link Result} of the query
	 */
	public Result run(CqnSelect select, Map<String, Object> namedValues);

	/**
	 * Executes a {@link CqnInsert} statement against the {@link CqnService}.
	 * @param insert the {@link CqnInsert} to be executed
	 * @return the {@link Result} of the insert statement
	 */
	public Result run(CqnInsert insert);

	/**
	 * Executes a {@link CqnUpsert} statement against the {@link CqnService}.
	 * @param upsert the {@link CqnUpsert} to be executed
	 * @return the {@link Result} of the upsert statement
	 */
	public Result run(CqnUpsert upsert);

	/**
	 * Executes a {@link CqnUpdate} statement with optional positional parameter values against the {@link CqnService}.
	 * @param update the {@link CqnUpdate} to be executed
	 * @param paramValues the optional positional parameter values
	 * @return the {@link Result} of the update statement
	 */
	public Result run(CqnUpdate update, Object... paramValues);

	/**
	 * Executes a {@link CqnUpdate} statement with named parameter values against the {@link CqnService}.
	 * @param update the {@link CqnUpdate} to be executed
	 * @param namedValues the named parameter values
	 * @return the {@link Result} of the update statement
	 */
	public Result run(CqnUpdate update, Map<String, Object> namedValues);

	/**
	 * Executes a {@link CqnUpdate} statement as batch with the given named parameter values against the {@link CqnService}.
	 * @param update the {@link CqnUpdate} to be executed
	 * @param valueSets the named parameter values
	 * @return the {@link Result} of the update statement
	 */
	public Result run(CqnUpdate update, Iterable<Map<String, Object>> valueSets);

	/**
	 * Executes a {@link CqnDelete} statement with optional positional parameter values against the {@link CqnService}.
	 * @param delete the {@link CqnDelete} to be executed
	 * @param paramValues the optional positional parameter values
	 * @return the {@link Result} of the delete statement
	 */
	public Result run(CqnDelete delete, Object... paramValues);

	/**
	 * Executes a {@link CqnDelete} statement with named parameter values against the {@link CqnService}.
	 * @param delete the {@link CqnDelete} to be executed
	 * @param namedValues the named parameter values
	 * @return the {@link Result} of the delete statement
	 */
	public Result run(CqnDelete delete, Map<String, Object> namedValues);

	/**
	 * Executes a {@link CqnDelete} statement as batch with the given named parameter values against the {@link CqnService}.
	 * @param delete the {@link CqnDelete} to be executed
	 * @param valueSets the named parameter values
	 * @return the {@link Result} of the delete statement
	 */
	public Result run(CqnDelete delete, Iterable<Map<String, Object>> valueSets);

}
