/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.draft;

import java.util.Map;

import com.sap.cds.Result;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.services.cds.ApplicationService;

/**
 * An {@link ApplicationService} that handles draft enabled entities.
 */
public interface DraftService extends ApplicationService {

	static String EVENT_DRAFT_EDIT = "draftEdit";
	static String EVENT_DRAFT_PREPARE = "draftPrepare";
	static String EVENT_DRAFT_SAVE = "draftActivate";
	static String EVENT_DRAFT_NEW = "DRAFT_NEW";
	static String EVENT_DRAFT_PATCH = "DRAFT_PATCH";
	static String EVENT_DRAFT_CANCEL = "DRAFT_CANCEL";
	static String EVENT_DRAFT_GC = "DRAFT_GC";

	// for event handlers only
	static String EVENT_ACTIVE_READ = "ACTIVE_READ";
	static String EVENT_DRAFT_READ = "DRAFT_READ";
	static String EVENT_DRAFT_CREATE = "DRAFT_CREATE";

	/**
	 * Saves a draft. That is, in case no active entity exists it will be created from the draft.
	 * Otherwise, it will be overwritten by the draft and the draft entity will be deleted.
	 * @param select the select statement specifying the entity to save
	 * @param paramValues the optional positional parameter values
	 * @return the {@link Result}
	 */
	Result saveDraft(CqnSelect select, Object... paramValues);

	/**
	 * Saves a draft. That is, in case no active entity exists will be created from the draft.
	 * Otherwise, it will be overwritten by the draft and the draft entity will be deleted.
	 * @param select the select statement specifying the entity to save
	 * @param namedValues the named parameter values
	 * @return the {@link Result}
	 */
	Result saveDraft(CqnSelect select, Map<String, Object> namedValues);

	/**
	 * Executes  checks to validate the current draft entity.
	 * @param select the statement specifying the draft to prepare
	 * @param sideEffectsQualifier the side effects qualifier
	 * @param paramValues the optional positional parameter values
	 * @return the {@link Result}
	 */
	Result prepareDraft(CqnSelect select, String sideEffectsQualifier, Object... paramValues);

	/**
	 * Executes  checks to validate the current draft entity.
	 * @param select the statement specifying the draft to prepare
	 * @param sideEffectsQualifier the side effects qualifier
	 * @param namedValues the named parameter values
	 * @return the {@link Result}
	 */
	Result prepareDraft(CqnSelect select, String sideEffectsQualifier, Map<String, Object> namedValues);

	/**
	 * Creates a new draft entity from an active entity.
	 * @param select the statement specifying the active entity to edit
	 * @param preserveChanges if {@code true} changes will be preserved
	 * @param paramValues the optional positional parameter values
	 * @return the {@link Result}
	 */
	Result editDraft(CqnSelect select, boolean preserveChanges, Object... paramValues);

	/**
	 * Creates a new draft entity from an active entity.
	 * @param select the statement specifying the active entity to edit
	 * @param preserveChanges if {@code true} changes will be preserved
	 * @param namedValues the named parameter values
	 * @return the {@link Result}
	 */
	Result editDraft(CqnSelect select, boolean preserveChanges, Map<String, Object> namedValues);

	/**
	 * Creates a new draft entity by executing the {@code insert} statement.
	 * @param insert the statement to execute
	 * @return the {@link Result} of the insert
	 */
	Result newDraft(CqnInsert insert);

	/**
	 * Updates draft entities by executing the {@code update} statement.
	 * @param update the statement to execute
	 * @param paramValues the optional positional parameter values
	 * @return the {@link Result} of the update
	 */
	Result patchDraft(CqnUpdate update, Object... paramValues);

	/**
	 * Updates draft entities by executing the {@code update} statement.
	 * @param update the statement to execute
	 * @param namedValues the named parameter values
	 * @return the {@link Result} of the update
	 */
	Result patchDraft(CqnUpdate update, Map<String, Object> namedValues);

	/**
	 * Updates draft entities by executing the {@code update} statement.
	 * @param update the statement to execute
	 * @param valueSets the named parameter values
	 * @return the {@link Result} of the update
	 */
	Result patchDraft(CqnUpdate update, Iterable<Map<String, Object>> valueSets);

	/**
	 * Deletes draft entities by executing the provided {@code delete} statement.
	 * @param delete the {@link CqnDelete} to be executed
	 * @param paramValues the optional positional parameter values
	 * @return the {@link Result} of the delete
	 */
	Result cancelDraft(CqnDelete delete, Object... paramValues);

	/**
	 * Deletes draft entities by executing the provided {@code delete} statement.
	 * @param delete the {@link CqnDelete} to be executed
	 * @param namedValues the named parameter values
	 * @return the {@link Result} of the delete
	 */
	Result cancelDraft(CqnDelete delete, Map<String, Object> namedValues);

	/**
	 * Deletes draft entities by executing the provided {@code delete} statement.
	 * @param delete the statement to execute
	 * @param valueSets the named parameter values
	 * @return the {@link Result} of the delete
	 */
	Result cancelDraft(CqnDelete delete, Iterable<Map<String, Object>> valueSets);

	/**
	 * This method deletes all drafts that exist longer than the draft deletion timeout ({@code cds.drafts.deletionTimeout}).
	 * @return the {@link Result} containing the number of deleted entries
	 */
	Result gcDrafts();

}
