/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.messages;

import java.util.List;
import java.util.Locale;
import java.util.function.Function;

import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.CoreFactory;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;

/**
 * {@code Message}s can be used to pass additional information along with the response.
 * Some clients such as Fiori UIs provide controls to show these messages in a user-friendly way.
 */
public interface Message {

	/**
	 * This method creates a standalone instance of {@link Message} that is not added to the {@link Messages} of the active {@link RequestContext}.
	 *
	 * @param severity the {@link Severity} of the {@link Message}
	 * @param text the final text of the message. For localization use {@link CdsRuntime#getLocalizedMessage(String, Object[], Locale)} explicitly.
	 * @return Created {@link Message} instance.
	 */
	static Message create(Severity severity, String text) {
		return CoreFactory.INSTANCE.createMessage(severity, text);
	}

	/**
	 * This method creates a standalone instance of {@link Message} that is not added to the {@link Messages} of the active {@link RequestContext}.
	 *
	 * @param severity the {@link Severity} of the {@link Message}
	 * @param text the final text of the message. For localization use {@link CdsRuntime#getLocalizedMessage(String, Object[], Locale)} explicitly.
	 * @param message the {@link Message} to copy optional arguments from
	 * @return the {@link Message} instance
	 */
	static Message create(Severity severity, String text, Message message) {
		return CoreFactory.INSTANCE.createMessage(severity, text, message);
	}

	/**
	 * This method creates a standalone instance of {@link Message} that is not added to the {@link Messages} of the active {@link RequestContext}.
	 *
	 * @param text the final text of the message. For localization use {@link CdsRuntime#getLocalizedMessage(String, Object[], Locale)} explicitly.
	 * @param exception the {@link ServiceException} to copy optional arguments from
	 * @return the {@link Message} instance
	 */
	static Message create(String text, ServiceException exception) {
		return CoreFactory.INSTANCE.createMessage(text, exception);
	}

	/**
	 * The localized message
	 * @return The localized message
	 */
	String getMessage();

	/**
	 * The {@link MessageLookup}
	 * @return The {@link MessageLookup}
	 */
	MessageLookup getMessageLookup();

	/**
	 * The optional code of the message
	 * @return The code
	 */
	String getCode(); // machine-readable

	/**
	 * The optional {@link MessageTarget}
	 * @return	The {@link MessageTarget}
	 */
	MessageTarget getTarget(); // Optional

	/**
	 * The optional list of additional {@link MessageTarget} instances
	 * @return The list of additional MessageTarget instances
	 */
	List<MessageTarget> getAdditionalTargets();

	/**
	 * The optional long text url
	 * @return	The long text url
	 */
	String getLongTextUrl(); // Optional

	/**
	 * Returns {@code true}, if the transition indicator is set for this message.
	 *
	 * A transition message is a request-specific message, that is valid during the runtime of a request.
	 * It might not reoccur if the same request is processed again.
	 * A state message has a relation to an entity document.
	 * It is typically a validation (error) message.
	 *
	 * @return {@code true}, if the transition indicator is set for this message
	 */
	boolean isTransition();

	/**
	 * The severity of the message.
	 * @return	The severity
	 */
	Severity getSeverity();

	/**
	 * Severity levels.
	 */
	enum Severity {

		SUCCESS (1), // Success - no action required

		INFO (2), // Information - no action required

		WARNING (3), // Warning - action may be required

		ERROR (4); // Error - action is required

		private final int numericSeverity;

		private Severity(int numericSeverity) {
			this.numericSeverity = numericSeverity;
		}

		public int getNumericSeverity() {
			return numericSeverity;
		}

	}

	/**
	 * Adds the passed {@code longTextUrl} to the current message.
	 *
	 * @param longTextUrl	The long text url to be set
	 * @return	The current message
	 */
	Message longTextUrl(String longTextUrl);

	/**
	 * Adds the passed {@code code} to the current message.
	 *
	 * @param code	The code to be set
	 * @return	The current message
	 */
	Message code(String code);

	/**
	 * Sets the {@code transition} indicator of the current message.
	 *
	 * A transition message is a request-specific message, that is valid during the runtime of a request.
	 * It might not reoccur if the same request is processed again.
	 * A state message has a relation to an entity document.
	 * It is typically a validation (error) message.
	 *
	 * @param isTransition {@code true}, if the transition indicator should be set, {@code false}, if it should be unset
	 * @return The current message
	 */
	Message transition(boolean isTransition);

	/**
	 * Sets the provided {@link MessageTarget}
	 *
	 * @param target the {@link MessageTarget}
	 * @return The current message
	 */
	Message target(MessageTarget target);

	/**
	 * Sets the provided string-based target. No further processing of the string is performed.
	 *
	 * @param target the string-based target
	 * @return	The current message
	 */
	default Message target(String target) {
		return target(MessageTarget.create(target));
	}

	/**
	 * Adds the passed path as target to the current {@link Message}.
	 * <p>
	 * The path is interpreted relative to the CQN statement, which was determined from the request.
	 * For CRUD events this CQN statement points to the targeted entity.
	 * For bound actions or functions this CQN statement points to the bound entity.
	 * <p>
	 * Is equivalent to calling {@code target(MessageTarget.PARAMETER_CQN, path)}
	 *
	 * @param path the path to the target element or association
	 * @return The current message
	 */
	default Message target(Function<StructuredType<?>, Object> path) {
		return target(MessageTarget.create(path));
	}

	/**
	 * Adds the passed target parameter and path as target to the current
	 * {@link Message}.
	 *
	 * @param parameter target parameter serving as the entry point for the path resolution.
	 *                  Passing {@link MessageTarget#PARAMETER_CQN} indicates that the path
	 *                  should be interpreted relatively to the target entity of the request.
	 *                  Alternatively you can pass names of action or function parameters.
	 *
	 * @param path      the path to the target element or association
	 * @return The current message
	 */
	default Message target(String parameter, Function<StructuredType<?>, Object> path) {
		return target(MessageTarget.create(parameter, path));
	}

	/**
	 * Adds the passed path as target to the current {@link Message}.
	 * This method allows to build the path in a type-safe way, by passing the corresponding entity or structured type interface.
	 * <p>
	 * The path is interpreted relative to the CQN statement, which was determined from the request.
	 * For CRUD events this CQN statement points to the targeted entity.
	 * For bound actions or functions this CQN statement points to the bound entity.
	 * <p>
	 * Is equivalent to calling {@code target(MessageTarget.PARAMETER_CQN, type, path)}
	 *
	 * @param type the root type of the path. Either an entity or a structured type.
	 * @param path the path to the target element or association
	 * @param <E>  the type of the root
	 * @return The current message
	 */
	default <E extends StructuredType<E>> Message target(Class<E> type, Function<E, Object> path) {
		return target(MessageTarget.create(type, path));
	}

	/**
	 * Adds the passed target parameter and path as target to the current
	 * {@link Message}. This method allows to build the path in a type-safe way,
	 * by passing the corresponding entity or structured type interface.
	 *
	 * @param parameter target parameter serving as the entry point for the path resolution.
	 *                  Passing {@link MessageTarget#PARAMETER_CQN} indicates that the path
	 *                  should be interpreted relatively to the target entity of the request.
	 *                  Alternatively you can pass names of action or function parameters.
	 *
	 * @param type the root type of the path. Either an entity or a structured type.
	 * @param path the path to the target element or association
	 * @param <E>  the type of the root
	 * @return The current message
	 */
	default <E extends StructuredType<E>> Message target(String parameter, Class<E> type, Function<E, Object> path) {
		return target(MessageTarget.create(parameter, type, path));
	}

	/**
	 * Adds the passed path and element as target to the current {@link Message}.
	 * <p>
	 * The path is interpreted relative to the CQN statement, which was determined from the request.
	 * For CRUD events this CQN statement points to the targeted entity.
	 * For bound actions or functions this CQN statement points to the bound entity.
	 * <p>
	 * This method can be used with the {@link com.sap.cds.CdsDataProcessor}
	 * and its functional interfaces.
	 *
	 * @param path    the path
	 * @param element the target element or association
	 * @return the current message
	 */
	default Message target(Path path, CdsElement element) {
		return target(MessageTarget.create(path, element));
	}

	/**
	 * Sets the provided list of {@link MessageTarget} instances as additional targets.
	 *
	 * @param additionalTargets the list of additional {@link MessageTarget} instances.
	 * @return the current message
	 */
	Message additionalTargets(List<MessageTarget> additionalTargets);

	/**
	 * Sets the provided array of {@link MessageTarget} instances as additional targets.
	 *
	 * @param additionalTargets the array of additional {@link MessageTarget} instances.
	 * @return the current message
	 */
	default Message additionalTargets(MessageTarget... additionalTargets) {
		return additionalTargets(List.of(additionalTargets));
	}

}
