/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.messages;

import java.util.function.Function;

import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.CoreFactory;

public interface MessageTarget {

	/**
     * Constant specifying a message target parameter, referring to the CQN of an
     * event context.
     * <p>
     * This is used to decide whether a binding parameter is required for the
     * message target output.
     */
    String PARAMETER_CQN = "cqn";

	/**
	 * Creates a string-based {@link MessageTarget}.
	 * No further processing of the string is performed.
	 *
	 * @param target the string-based target
	 * @return	The created {@link MessageTarget}
	 */
	static MessageTarget create(String target) {
		return CoreFactory.INSTANCE.createMessageTarget(target);
	}

	/**
	 * Creates a {@link MessageTarget} based on a CQN path.
	 * <p>
	 * The path is interpreted relative to the CQN statement, which was determined from the request.
	 * For CRUD events this CQN statement points to the targeted entity.
	 * For bound actions or functions this CQN statement points to the bound entity.
	 * <p>
	 * Is equivalent to calling {@code create(MessageTarget.PARAMETER_CQN, path)}
	 *
	 * @param path the path to the target element or association
	 * @return The created {@link MessageTarget}
	 */
	static MessageTarget create(Function<StructuredType<?>, Object> path) {
		return create(PARAMETER_CQN, path);
	}

	/**
	 * Creates a {@link MessageTarget} based on a parameter and a CQN path.
	 *
	 * @param parameter target parameter serving as the entry point for the path resolution.
	 *                  Passing {@link MessageTarget#PARAMETER_CQN} indicates that the path
	 *                  should be interpreted relatively to the target entity of the request.
	 *                  Alternatively you can pass names of action or function parameters.
	 *
	 * @param path      the path to the target element or association
	 * @return The created {@link MessageTarget}
	 */
	static MessageTarget create(String parameter, Function<StructuredType<?>, Object> path) {
		return CoreFactory.INSTANCE.createMessageTarget(parameter, path);
	}

	/**
	 * Creates a {@link MessageTarget} based on a CQN path.
	 * This method allows to build the path in a type-safe way, by passing the corresponding entity or structured type interface.
	 * <p>
	 * The path is interpreted relative to the CQN statement, which was determined from the request.
	 * For CRUD events this CQN statement points to the targeted entity.
	 * For bound actions or functions this CQN statement points to the bound entity.
	 * <p>
	 * Is equivalent to calling {@code create(MessageTarget.PARAMETER_CQN, type, path)}
	 *
	 * @param type the root type of the path. Either an entity or a structured type.
	 * @param path the path to the target element or association
	 * @param <E>  the type of the root
	 * @return The created {@link MessageTarget}
	 */
	static <E extends StructuredType<E>> MessageTarget create(Class<E> type, Function<E, Object> path) {
		return create(PARAMETER_CQN, type, path);
	}

	/**
	 * Creates a {@link MessageTarget} based on a parameter and a CQN path.
	 * This method allows to build the path in a type-safe way,
	 * by passing the corresponding entity or structured type interface.
	 *
	 * @param parameter target parameter serving as the entry point for the path resolution.
	 *                  Passing {@link MessageTarget#PARAMETER_CQN} indicates that the path
	 *                  should be interpreted relatively to the target entity of the request.
	 *                  Alternatively you can pass names of action or function parameters.
	 *
	 * @param type the root type of the path. Either an entity or a structured type.
	 * @param path the path to the target element or association
	 * @param <E>  the type of the root
	 * @return The created {@link MessageTarget}
	 */
	static <E extends StructuredType<E>> MessageTarget create(String parameter, Class<E> type, Function<E, Object> path) {
		return CoreFactory.INSTANCE.createMessageTarget(parameter, type, path);
	}

	/**
	 * Creates a {@link MessageTarget} based on a {@link Path} and a {@link CdsElement}.
	 * <p>
	 * The path is interpreted relative to the CQN statement, which was determined from the request.
	 * For CRUD events this CQN statement points to the targeted entity.
	 * For bound actions or functions this CQN statement points to the bound entity.
	 * <p>
	 * This method can be used with the {@link com.sap.cds.CdsDataProcessor}
	 * and its functional interfaces.
	 *
	 * @param path    the path
	 * @param element the target element or association
	 * @return The created {@link MessageTarget}
	 */
	static MessageTarget create(Path path, CdsElement element) {
		return CoreFactory.INSTANCE.createMessageTarget(path, element);
	};

	/**
	 * Creates a {@link MessageTarget} based on a parameter and a {@link CqnReference}
	 *
	 * @param parameter target parameter serving as the entry point for the path resolution.
	 *                  Passing {@link MessageTarget#PARAMETER_CQN} indicates that the path
	 *                  should be interpreted relatively to the target entity of the request.
	 *                  Alternatively you can pass names of action or function parameters.
	 * @param ref		The reference to the target element or association
	 * @return The created {@link MessageTarget}
	 */
	static MessageTarget create(String parameter, CqnReference ref) {
		return CoreFactory.INSTANCE.createMessageTarget(parameter, ref);
	}

	/**
	 * @return the CQN reference of the message target
	 */
	CqnReference getRef();

	/**
	 * @return (target) parameter of the message target
	 */
	String getParameter();

}
