/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.outbox;

import com.sap.cds.services.CoreFactory;
import com.sap.cds.services.Service;
import com.sap.cds.services.cds.AsyncCqnService;

/**
 * Defines the outbox service API. The outbox can be used to couple emitting of
 * events on a service to the currently active transaction.
 *
 * The following applies when using the outbox:
 * - The outbox message is stored in the outbox.
 * - If the transaction fails, the outbox message is not stored in the outbox and is never emitted.
 * - A persisted outbox message is emitted eventually (asynchronously or synchronously, depending on the outbox implementation)
 *
 */
public interface OutboxService extends Service {

	public static final String PERSISTENT_ORDERED_NAME = "DefaultOutboxOrdered";
	public static final String PERSISTENT_UNORDERED_NAME = "DefaultOutboxUnordered";

	public static final String INMEMORY_NAME = "OutboxService$InMemory";

	/**
	 * Returns the underlying service of the outbox wrapped service.
	 * The returned service instance is the underlying outbox-independent instance executing synchronously.
	 *
	 * @param <S> the type of service
	 * @param service the service to unwrap from outbox wrapping
	 * @return the underlying service of the outbox wrapped service.
	 */
	static <S extends Service> S unboxed(S service) {
		return CoreFactory.INSTANCE.unboxed(service);
	}

	/**
	 * Stores the given {@link OutboxMessage} under the given outbox event in the outbox.
	 *
	 * @param outboxEvent the outbox event with which the message is stored and again published by the {@link OutboxService}
	 * @param message the {@link OutboxMessage} stored and again published by the {@link OutboxService}
	 */
	void submit(String outboxEvent, OutboxMessage message);

	/**
	 * Returns a wrapper around the given service that executes asynchronously by submitting all events to this outbox instance.
	 * The returned instance can be cached, reused and is thread-safe.
	 *
	 * If the passed service is already wrapped with this outbox, the passed service is returned.
	 * If the passed service is wrapped by another outbox, a new wrapper for the underlying service is returned.
	 *
	 * @param <S> the type of service
	 * @param service the service to be outboxed
	 * @return a wrapper around the given service that executes asynchronously by submitting all events to this outbox instance.
	 */
	<S extends Service> S outboxed(S service);

	/**
	 * Returns a wrapper around the given service that executes asynchronously by submitting all events to this outbox instance.
	 * The returned instance can be cached, reused and is thread-safe.
	 * <p>
	 * The given service is wrapped with an asynchronous interface, matching the original interface of the service,
	 * but providing all methods with a <code>void</code> return type. See {@link AsyncCqnService} for an example.
	 * <p>
	 * If the passed service is already wrapped with this outbox, the passed service is returned.
	 * If the passed service is wrapped by another outbox, a new wrapper for the underlying service is returned.
	 *
	 * @param <A> the type of the async interface
	 * @param service the service to be outboxed
	 * @param asyncInterface the class of the async interface
	 * @return a wrapper around the given service that executes asynchronously by submitting all events to this outbox instance.
	 */
	<A extends Service> A outboxed(Service service, Class<A> asyncInterface);

}
