/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.runtime;

import java.util.function.Consumer;
import java.util.function.Function;

import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.services.changeset.ChangeSetContext;

public interface ChangeSetContextRunner {

	/**
	 * Eagerly marks the {@link ChangeSetContext} as transactional (see {@link ChangeSetContext#markTransactional()}).
	 * <p>
	 * Transactional {@link ChangeSetContext}s ensure to allocate transactions and corresponding resources (e.g. SQL connections),
	 * upon first interaction with a transactional data sources. Those resources are released only after closing the {@link ChangeSetContext}.
	 * A transactional data source might mark the {@link ChangeSetContext} as transactional, if it requires a transaction
	 * (e.g. when executing a {@link CqnInsert}, {@link CqnUpdate}, {@link CqnDelete} or {@link CqnSelect} with locks).
	 * <p>
	 * {@link ChangeSetContext} that are not marked transactional allow transactional data sources to free resources earlier,
	 * e.g. after each interaction. For example in case of JDBC data sources, SQL connections
	 * can be returned to the connection pool after executing a {@link CqnSelect} without locks.
	 * In that case they are not required to be kept for the entire {@link ChangeSetContext} scope.
	 * <p>
	 * Once a {@link ChangeSetContext} is marked as transactional it can't be unmarked again.
	 *
	 * @return the {@link ChangeSetContextRunner}
	 */
	ChangeSetContextRunner markTransactional();

	/**
	 * Opens a {@link ChangeSetContext} and runs the given handler within its scope.
	 * The {@code ChangeSetContext} marks a scope in which all opened transactions
	 * etc. are committed cancelled together. Note that single transactions may fail
	 * during commit but others might be successful.
	 * <p>
	 * The {@code ChangeSetContext} will be propagated to all
	 * {@link com.sap.cds.services.EventContext} instances in the service call
	 * hierarchy.
	 *
	 * @param <T>              The type of the response
	 * @param changeSetHandler The handler for processing the request within the context
	 * @return A generic response object of the handler
	 */
	<T> T run(Function<ChangeSetContext, T> changeSetHandler);

	/**
	 * Opens a {@link ChangeSetContext} and runs the given handler within its scope.
	 * The {@code ChangeSetContext} marks a scope in which all opened transactions
	 * etc. are committed cancelled together. Note that single transactions may fail
	 * during commit but others might be successful.
	 * <p>
	 * The {@code ChangeSetContext} will be propagated to all
	 * {@link com.sap.cds.services.EventContext} instances in the service call
	 * hierarchy.
	 *
	 * @param changeSetHandler The handler for processing the request within the context
	 */
	void run(Consumer<ChangeSetContext> changeSetHandler);

}
