/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.handler.Handler;
import com.sap.cds.services.handler.annotations.HandlerOrder;

/**
 * The {@link Service} is the root consumption API that every {@link Service} instance offers. It
 * offers the generic {@link #emit(EventContext)} method, which is also the basis of more
 * specialized consumption API methods. <br>
 * To enable the Provider API the {@link Service} interface also offers programmatic registration of
 * {@link Handler handlers} to handle events. {@link Handler Handlers} can be registered for the
 * three phases BEFORE, ON and AFTER using events and entities, which should be handled. <br>
 * The BEFORE phase is useful for doing filtering, validation and other types of pre-processing of
 * the input parameters of an event. {@link Handler} are called one after the other and can abort
 * the complete processing of the event at any time by throwing an exception. If during the BEFORE
 * phase the {@link EventContext#setCompleted()} method is called, the event processing directly
 * jumps to the AFTER phase. If no {@link Handler} completes the {@link EventContext} during the
 * BEFORE phase, or does not abort the event processing by throwing an exception, the ON phase is
 * started. <br>
 * The ON phase is meant to implement the core processing of the event. It's {@link Handler} should
 * compute the output parameters based in the input parameters of the event. {@link Handler} are
 * called one after the other. If a {@link Handler} performs any action during the ON phase, it
 * should call {@link EventContext#setCompleted()} to indicate that the event was successfully
 * processed. As soon as the {@link EventContext} is completed, the ON phase is finished and the
 * event processing jumps to the AFTER phase. In case a {@link Handler} throws an exception the
 * event processing is aborted. If no {@link Handler} completes the {@link EventContext} during the
 * ON phase, it is treated as an error and an exception is thrown. <br>
 * The AFTER phase is useful for doing post-processing of the output parameters of an event. {@link
 * Handler} are called one after the other. A {@link Handler} in this phase can still abort the
 * event processing by throwing an exception. No further {@link Handler} are called in this case.
 * Alternatively it can also call {@link ChangeSetContext#markForCancel()}, which cancels the active
 * {@link ChangeSetContext} and causes a rollback of transactions. However it does not abort the
 * event processing itself.
 */
public interface Service {

  /**
   * Creates a new {@link Service}
   *
   * @param name the name of the service
   * @return the {@link Service}
   */
  static Service create(String name) {
    return CoreFactory.INSTANCE.createService(name);
  }

  // PROVIDER API
  /**
   * Registers a {@link Handler} on this {@link Service} for the BEFORE phase of the processing of
   * an {@link EventContext}.
   *
   * @param events the events the custom handler is registered for
   * @param entities the entities the custom handler is registered for
   * @param order the order of the handler. the lower the order the earlier the handler gets called.
   *     if two handlers have the same order they get ordered in their registration order
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the BEFORE phase.
   */
  public void before(String[] events, String[] entities, int order, Handler handler);

  /**
   * Registers a {@link Handler} on this {@link Service} for the BEFORE phase of the processing of
   * an {@link EventContext}.
   *
   * @param events the events the custom handler is registered for
   * @param entities the entities the custom handler is registered for
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the BEFORE phase.
   */
  public default void before(String[] events, String[] entities, Handler handler) {
    before(events, entities, HandlerOrder.DEFAULT, handler);
  }

  /**
   * Registers a {@link Handler} on this {@link Service} for the BEFORE phase of the processing of
   * an {@link EventContext}.
   *
   * @param event the event the custom handler is registered for
   * @param entity the entity the custom handler is registered for
   * @param order the order of the handler. the lower the order the earlier the handler gets called.
   *     if two handlers have the same order they get ordered in their registration order
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the BEFORE phase.
   */
  public default void before(String event, String entity, int order, Handler handler) {
    before(
        event == null ? null : new String[] {event},
        entity == null ? null : new String[] {entity},
        order,
        handler);
  }

  /**
   * Registers a {@link Handler} on this {@link Service} for the BEFORE phase of the processing of
   * an {@link EventContext}.
   *
   * @param event the event the custom handler is registered for
   * @param entity the entity the custom handler is registered for
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the BEFORE phase.
   */
  public default void before(String event, String entity, Handler handler) {
    before(event, entity, HandlerOrder.DEFAULT, handler);
  }

  /**
   * Registers a {@link Handler} on this {@link Service} for the ON phase of the processing of an
   * {@link EventContext}.
   *
   * @param events the events the custom handler is registered for
   * @param entities the entities the custom handler is registered for
   * @param order the order of the handler. the lower the order the earlier the handler gets called.
   *     if two handlers have the same order they get ordered in their registration order
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the ON phase.
   */
  public void on(String[] events, String[] entities, int order, Handler handler);

  /**
   * Registers a {@link Handler} on this {@link Service} for the ON phase of the processing of an
   * {@link EventContext}.
   *
   * @param events the events the custom handler is registered for
   * @param entities the entities the custom handler is registered for
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the ON phase.
   */
  public default void on(String[] events, String[] entities, Handler handler) {
    on(events, entities, HandlerOrder.DEFAULT, handler);
  }

  /**
   * Registers a {@link Handler} on this {@link Service} for the ON phase of the processing of an
   * {@link EventContext}.
   *
   * @param event the event the custom handler is registered for
   * @param entity the entity the custom handler is registered for
   * @param order the order of the handler. the lower the order the earlier the handler gets called.
   *     if two handlers have the same order they get ordered in their registration order
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the ON phase.
   */
  public default void on(String event, String entity, int order, Handler handler) {
    on(
        event == null ? null : new String[] {event},
        entity == null ? null : new String[] {entity},
        order,
        handler);
  }

  /**
   * Registers a {@link Handler} on this {@link Service} for the ON phase of the processing of an
   * {@link EventContext}.
   *
   * @param event the event the custom handler is registered for
   * @param entity the entity the custom handler is registered for
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the ON phase.
   */
  public default void on(String event, String entity, Handler handler) {
    on(event, entity, HandlerOrder.DEFAULT, handler);
  }

  /**
   * Registers a {@link Handler} on this {@link Service} for the AFTER phase of the processing of an
   * {@link EventContext}.
   *
   * @param events the events the custom handler is registered for
   * @param entities the entities the custom handler is registered for
   * @param order the order of the handler. the lower the order the earlier the handler gets called.
   *     if two handlers have the same order they get ordered in their registration order
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the AFTER phase.
   */
  public void after(String[] events, String[] entities, int order, Handler handler);

  /**
   * Registers a {@link Handler} on this {@link Service} for the AFTER phase of the processing of an
   * {@link EventContext}.
   *
   * @param events the events the custom handler is registered for
   * @param entities the entities the custom handler is registered for
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the AFTER phase.
   */
  public default void after(String[] events, String[] entities, Handler handler) {
    after(events, entities, HandlerOrder.DEFAULT, handler);
  }

  /**
   * Registers a {@link Handler} on this {@link Service} for the AFTER phase of the processing of an
   * {@link EventContext}.
   *
   * @param event the event the custom handler is registered for
   * @param entity the entity the custom handler is registered for
   * @param order the order of the handler. the lower the order the earlier the handler gets called.
   *     if two handlers have the same order they get ordered in their registration order
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the AFTER phase.
   */
  public default void after(String event, String entity, int order, Handler handler) {
    after(
        event == null ? null : new String[] {event},
        entity == null ? null : new String[] {entity},
        order,
        handler);
  }

  /**
   * Registers a {@link Handler} on this {@link Service} for the AFTER phase of the processing of an
   * {@link EventContext}.
   *
   * @param event the event the custom handler is registered for
   * @param entity the entity the custom handler is registered for
   * @param handler the {@link Handler} implementing the business-logic, which should become active
   *     during the AFTER phase.
   */
  public default void after(String event, String entity, Handler handler) {
    after(event, entity, HandlerOrder.DEFAULT, handler);
  }

  // GENERIC CONSUMPTION API
  /**
   * The generic consumption API to emit arbitrary events, represented by {@link EventContext}
   * objects. More specialized consumption APIs (for example {@link CqnService}) are implemented as
   * a wrapper around this method. <br>
   * Emitting an {@link EventContext} will start the processing of the event and calls the
   * registered {@link Handler} instances. If {@link #emit(EventContext)} is called outside of an
   * active {@link ChangeSetContext}, an internal {@link ChangeSetContext} is opened and properly
   * closed. {@link Handler} can therefore rely on an active {@link ChangeSetContext}. <br>
   * The {@link EventContext} passed to this method, should be propagated with the required input
   * parameters of the event. After the method finished, the respective output parameters can be
   * read from the passed in {@link EventContext}.
   *
   * @param context the {@link EventContext} to be emitted
   */
  public void emit(EventContext context);

  /**
   * Returns the name of the {@link Service}. The name might be the fully qualified name of a
   * service, which is also present in the {@link CdsModel}. If the service is a technical service,
   * the name is an arbitrary name, which is not reflected in the {@link CdsModel}.
   *
   * @return the name of the {@link Service}
   */
  public String getName();
}
