/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services;

import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.environment.CdsProperties.Errors;
import com.sap.cds.services.messages.MessageTarget;
import com.sap.cds.services.request.RequestContext;
import java.util.Locale;
import java.util.function.Function;

/**
 * Helper methods for creating (localized) messages in {@link ServiceException}.
 *
 * <p>{@code ServiceExceptionUtils#getMessage} is used to implement {@code
 * ServiceException#getMessage} and {@code ServiceExceptionUtils#getLocalizedMessage} is used to
 * implement {@code ServiceException#getLocalizedMessage} respectively.
 */
public interface ServiceExceptionUtils {

  /**
   * Creates a message based on code and args. {@code code} may either be a SL4J-message format or a
   * bundle key. The JVM's default locale is taken to lookup code in resource bundles.
   *
   * @param code The used code
   * @param args The arguments for message formats.
   * @return The created message
   */
  String getMessage(String code, Object[] args);

  /**
   * Creates a message based on code and args. {@code code} may either be a SL4J-message format or a
   * bundle key. The specified locale is taken to lookup the code in resource bundles. If the
   * specified locale is <code>null</code> the locale provided of the current {@link RequestContext}
   * is used
   *
   * @param code The used code
   * @param args The arguments for message formats.
   * @param locale The locale
   * @return The created message
   * @deprecated Use {@link #getLocalizedMessage(String, Object[], Locale, boolean)} instead.
   */
  @Deprecated(forRemoval = true)
  default String getLocalizedMessage(String code, Object[] args, Locale locale) {
    return getLocalizedMessage(code, args, locale, true);
  }

  /**
   * Creates a message based on code and args. {@code code} may either be a SL4J-message format or a
   * bundle key. The specified locale is taken to lookup the code in resource bundles. If the
   * specified locale is <code>null</code> the locale provided of the current {@link RequestContext}
   * is used
   *
   * @param code The used code
   * @param args The arguments for message formats.
   * @param locale The locale
   * @param errorStatusFallback If {@code true}, a fallback to the description is done for
   *     well-known {@link ErrorStatus}.
   * @return The created message
   */
  String getLocalizedMessage(
      String code, Object[] args, Locale locale, boolean errorStatusFallback);

  /**
   * @param target
   * @return
   * @deprecated Use {@link MessageTarget#create(String)} instead.
   */
  @Deprecated(forRemoval = true)
  default MessageTarget getMessageTarget(String target) {
    return MessageTarget.create(target);
  }

  /**
   * @param parameter
   * @param path
   * @return
   * @deprecated Use {@link MessageTarget#create(String, Function)} instead.
   */
  @Deprecated(forRemoval = true)
  default MessageTarget getMessageTarget(
      String parameter, Function<StructuredType<?>, Object> path) {
    return MessageTarget.create(parameter, path);
  }

  /**
   * @param <E>
   * @param parameter
   * @param type
   * @param path
   * @return
   * @deprecated Use {@link MessageTarget#create(String, Class, Function)} instead.
   */
  @Deprecated(forRemoval = true)
  default <E extends StructuredType<E>> MessageTarget getMessageTarget(
      String parameter, Class<E> type, Function<E, Object> path) {
    return MessageTarget.create(parameter, type, path);
  }

  /**
   * @param path
   * @param element
   * @return
   * @deprecated Use {@link MessageTarget#create(Path, CdsElement)} instead.
   */
  @Deprecated(forRemoval = true)
  default MessageTarget getMessageTarget(Path path, CdsElement element) {
    return MessageTarget.create(path, element);
  }

  /**
   * @return the active {@link Errors} properties
   */
  Errors getErrorsProperties();
}
