/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.draft;

import com.sap.cds.Result;
import com.sap.cds.ql.CdsName;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.EventName;
import java.util.Map;

/** The {@link EventContext} for the {@link DraftService#EVENT_DRAFT_PREPARE} event. */
@EventName(DraftService.EVENT_DRAFT_PREPARE)
public interface DraftPrepareEventContext extends EventContext {

  public static final String SIDE_EFFECTS_QUALIFIER = "SideEffectsQualifier";

  /**
   * Creates an {@link EventContext} already overlaid with this interface. The event is set to be
   * {@link DraftService#EVENT_DRAFT_PREPARE}
   *
   * @param entityName the name of the entity
   * @return the {@link DraftPrepareEventContext}
   */
  static DraftPrepareEventContext create(String entityName) {
    return EventContext.create(DraftPrepareEventContext.class, entityName);
  }

  @Override
  DraftService getService();

  // IN
  /**
   * @return the {@link CqnSelect} statement
   */
  CqnSelect getCqn();

  /**
   * Sets the {@link CqnSelect} statement
   *
   * @param cqn the {@link CqnSelect} statement
   */
  void setCqn(CqnSelect cqn);

  /**
   * Sets the sideEffectQualifier
   *
   * @param sideEffectsQualifier the side effects qualifier
   */
  @CdsName(SIDE_EFFECTS_QUALIFIER)
  void setSideEffectsQualifier(String sideEffectsQualifier);

  /**
   * @return the side effects qualifier
   */
  @CdsName(SIDE_EFFECTS_QUALIFIER)
  String getSideEffectsQualifier();

  /**
   * The CQN named values, for parameterized {@link CqnSelect} statements
   *
   * @return the CQN named values
   */
  Map<String, Object> getCqnNamedValues();

  /**
   * Sets the CQN named values, for parameterized {@link CqnSelect} statements
   *
   * @param namedValues the CQN named values
   */
  void setCqnNamedValues(Map<String, Object> namedValues);

  // OUT
  /**
   * @return the {@link Result} of the {@link DraftService#EVENT_DRAFT_PREPARE} event
   */
  Result getResult();

  /**
   * Sets the result of the {@link DraftService#EVENT_DRAFT_PREPARE} event
   *
   * @param result a {@link Result} or a list of Maps, which will be transformed into a {@link
   *     Result}
   */
  void setResult(Iterable<? extends Map<String, ?>> result);
}
