/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.request;

import com.sap.cds.services.CoreFactory;
import java.time.Instant;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;

/**
 * Interface to access request-specific information. For example, if the request is processed by a
 * HTTP-based protocol adapter these methods provide access to the HTTP request.
 */
public interface ParameterInfo {

  /**
   * Creates a {@link ModifiableUserInfo} based on default values of a clear {@link ParameterInfo}.
   *
   * @return The created {@link ModifiableUserInfo} instance.
   */
  static ModifiableParameterInfo create() {
    return CoreFactory.INSTANCE.createParameterInfo(null);
  }

  /**
   * @return the correlation id
   */
  default String getCorrelationId() {
    return null;
  }

  /**
   * Returns the value of a header with the given identifier. In case the header with {@code id} has
   * several values, the result contains the values comma-separated.
   *
   * @param id the header identifier
   * @return the value of the header
   */
  default String getHeader(String id) {
    return getHeaders().get(id);
  }

  /**
   * Returns the header values as {@link Map}. Note that this might be an expensive operation.
   *
   * @return the header values as {@link Map}.
   */
  default Map<String, String> getHeaders() {
    return Collections.emptyMap();
  }

  /**
   * Returns the value of a query parameter with the given key
   *
   * @param key the query parameter key
   * @return the value of the query parameter
   */
  default String getQueryParameter(String key) {
    return getQueryParams().get(key);
  }

  /**
   * Returns the query parameter values as {@link Map}. Note that this might be an expensive
   * operation.
   *
   * @return the query parameter values as {@link Map}.
   */
  default Map<String, String> getQueryParams() {
    return Collections.emptyMap();
  }

  /**
   * @return the preferred {@link Locale} set by the request
   */
  default Locale getLocale() {
    return null;
  }

  /**
   * @return parsed {@link Instant} value of request parameter "valid-from" or constructed from
   *     {@link Long#MIN_VALUE} if not available or parsing failed.
   */
  default Instant getValidFrom() {
    return null;
  }

  /**
   * @return parsed {@link Instant} value of request parameter "valid-to" or constructed from {@link
   *     Long#MAX_VALUE} if not available or parsing failed.
   */
  default Instant getValidTo() {
    return null;
  }

  /**
   * Creates a {@link ModifiableParameterInfo} based on this {@link ParameterInfo}.
   *
   * @return The created {@link ModifiableParameterInfo} instance.
   */
  default ModifiableParameterInfo copy() {
    return CoreFactory.INSTANCE.createParameterInfo(this);
  }
}
