/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.request;

import com.sap.cds.Struct;
import com.sap.cds.services.CoreFactory;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Interface to fetch user and authorization information.
 *
 * <p>The current user refers to the authenticated user for whom the request context has been
 * opened. All user information like ids, names, roles, tenant and user attributes are assumed to be
 * verified, i.e. they are secured by adequate means like JWT verification.
 */
public interface UserInfo {

  /**
   * Creates a {@link ModifiableUserInfo} based on default values of a clear {@link UserInfo}.
   *
   * @return The created {@link ModifiableUserInfo} instance.
   */
  static ModifiableUserInfo create() {
    return CoreFactory.INSTANCE.createUserInfo(null);
  }

  /**
   * Returns the verified user id of the request's authenticated user.
   *
   * @return The user id.
   */
  default String getId() {
    return null;
  }

  /**
   * Returns the verified user name of the request's authenticated user.
   *
   * @return The user name.
   */
  String getName();

  /**
   * Returns the verified tenant of the request's authenticated user.
   *
   * @return The tenant of the user.
   */
  default String getTenant() {
    return null;
  }

  /**
   * Returns {@code true} if the user is a system user and not a named user.
   *
   * @return {@code true} if the user is a system user and not a named user.
   */
  default boolean isSystemUser() {
    return false;
  }

  /**
   * Returns {@code true} if the user is the internal user.
   *
   * @return {@code true} if the user is the internal user.
   */
  default boolean isInternalUser() {
    return false;
  }

  /**
   * Returns {@code true} if {@link UserInfo} is filled with data of an authenticated user.
   *
   * @return {@code true} if {@link UserInfo} is filled with data of an authenticated user.
   */
  boolean isAuthenticated();

  /**
   * Returns {@code true} if {@link UserInfo} is a privileged user, i.e. passes all authorization
   * checks. Privileged users can only be used internally.
   *
   * @return {@code true} if {@link UserInfo} is a privileged user.
   */
  default boolean isPrivileged() {
    return false;
  }

  /**
   * Checks if the current user is associated with the specified user role.
   *
   * @param role The name of the role.
   * @return {@code true} if role is present, {@code false} otherwise.
   */
  default boolean hasRole(String role) {
    return getRoles().contains(role);
  }

  /**
   * Returns the {@link List} of granted user roles.
   *
   * @return the {@link List} of roles.
   */
  default Set<String> getRoles() {
    return Collections.emptySet();
  }

  /**
   * Returns a {@link List} of values of the specified attribute as presented in the request.
   *
   * @param attribute The name of the attribute.
   * @return The value of the attribute or an empty list if no values are present.
   */
  default List<String> getAttributeValues(String attribute) {
    List<String> result = getAttributes().get(attribute);
    return result != null ? result : Collections.emptyList();
  }

  /**
   * Returns all user attributes as a {@link Map}.
   *
   * @return The user attributes.
   */
  default Map<String, List<String>> getAttributes() {
    return Collections.emptyMap();
  }

  /**
   * Returns additional attribute with given name if existing.
   *
   * @param name The name of the attribute.
   * @return The attribute value as {@code Object}.
   */
  default Object getAdditionalAttribute(String name) {
    return getAdditionalAttributes().get(name);
  }

  /**
   * Returns all additional attributes as {@link Map}.
   *
   * @return The additional attributes,
   */
  default Map<String, Object> getAdditionalAttributes() {
    return Collections.emptyMap();
  }

  /**
   * Provides type-safe access to {@code UserInfo} and additional attributes. Creates a concrete
   * proxy instance implementing {@code UserInfo} and a sub interface {@code concreteUserInfoClazz}
   * which has a concrete API for the additional attributes according to the chosen authentication.
   *
   * <p>Example:
   *
   * <pre>XsuaaUserInfo xsuaaUser = eventContext.getUserInfo().as(XsuaaUserInfo.class);</pre>
   *
   * @param <T> the type of the concrete {@link UserInfo} interface.
   * @param concreteUserInfoClazz The sub interface providing the concrete API of additional
   *     attributes.
   * @return A concrete proxy instance implementing {@code UserInfo} and {@code
   *     concreteUserInfoClazz}.
   */
  default <T extends UserInfo> T as(Class<T> concreteUserInfoClazz) {
    return Struct.access(Collections.emptyMap()).as(concreteUserInfoClazz);
  }

  /**
   * Creates a {@link ModifiableUserInfo} based on this {@link UserInfo}.
   *
   * @return The created {@link ModifiableUserInfo} instance.
   */
  default ModifiableUserInfo copy() {
    return CoreFactory.INSTANCE.createUserInfo(this);
  }
}
