/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.runtime;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.CoreFactory;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.environment.ApplicationInfoProvider;
import com.sap.cds.services.environment.PropertiesProvider;
import com.sap.cds.services.environment.ServiceBindingProvider;
import com.sap.cds.services.messages.LocalizedMessageProvider;
import java.io.IOException;
import java.util.function.Supplier;

public interface CdsRuntimeConfigurer {

  /**
   * Creates a new {@link CdsRuntimeConfigurer}.
   *
   * <p>In addition it automatically adds all {@link CdsRuntimeConfiguration} modules registered
   * through {@link ExtendedServiceLoader} mechanisms.
   *
   * @return the {@link CdsRuntimeConfigurer}
   */
  static CdsRuntimeConfigurer create() {
    return CoreFactory.INSTANCE.createCdsRuntimeConfigurer(null);
  }

  /**
   * Creates a new {@link CdsRuntimeConfigurer} with the given {@link PropertiesProvider}.
   *
   * <p>In addition it automatically adds all {@link CdsRuntimeConfiguration} modules registered
   * through {@link ExtendedServiceLoader} mechanisms.
   *
   * @return the {@link CdsRuntimeConfigurer}
   */
  static CdsRuntimeConfigurer create(PropertiesProvider propertiesProvider) {
    return CoreFactory.INSTANCE.createCdsRuntimeConfigurer(propertiesProvider);
  }

  /**
   * The {@link CdsRuntime} that is configured by this {@link CdsRuntimeConfigurer}. It can be
   * retrieved at any time, although it might get enriched by subsequent calls to this {@link
   * CdsRuntimeConfigurer}.
   *
   * @return the {@link CdsRuntime}
   */
  CdsRuntime getCdsRuntime();

  /**
   * Adds a given {@link CdsRuntimeConfiguration} to the {@link CdsRuntimeConfigurer}. This
   * configuration will be applied in parts, when the respective bulk operations are triggered, for
   * example {@link #serviceConfigurations()} or {@link #eventHandlerConfigurations()}
   *
   * @param configuration the {@link CdsRuntimeConfiguration}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer configuration(CdsRuntimeConfiguration configuration);

  /**
   * Applies all environment configuration modules to the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer environmentConfigurations();

  /**
   * Sets the {@link ServiceBindingProvider} on the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @param provider the {@link ServiceBindingProvider}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer environment(ServiceBindingProvider provider);

  /**
   * Sets the {@link ApplicationInfoProvider} on the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @param provider the {@link ApplicationInfoProvider}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer environment(ApplicationInfoProvider provider);

  /**
   * Sets the default CDS model on the {@link CdsRuntime} of this {@link CdsRuntimeConfigurer}
   *
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer cdsModel();

  /**
   * Sets the CDS model from the specified CSN file path on the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @param csnPath the CSN file path
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer cdsModel(String csnPath);

  /**
   * Sets the given {@link CdsModel} on the {@link CdsRuntime} of this {@link CdsRuntimeConfigurer}
   *
   * @param model the {@link CdsModel}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer cdsModel(CdsModel model);

  /**
   * Applies all service configuration modules to the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer serviceConfigurations();

  /**
   * Registers a given {@link Service} in the {@link ServiceCatalog} of the {@link CdsRuntime}.
   *
   * @param service the {@link Service}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer service(Service service);

  /**
   * Applies all event handler configuration modules to the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer eventHandlerConfigurations();

  /**
   * Registers an event handler instance with the {@link CdsRuntime}.
   *
   * @param handler the event handler instance
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer eventHandler(Object handler);

  /**
   * Registers an event handler class with the {@link CdsRuntime}, given a factory to create
   * instances of the event handler
   *
   * @param <T> the event handler type
   * @param handlerClass the event handler class
   * @param handlerFactory the supplier for instances of the event handler
   * @return the {@link CdsRuntimeConfigurer}
   */
  <T> CdsRuntimeConfigurer eventHandler(Class<T> handlerClass, Supplier<T> handlerFactory);

  /**
   * Scans the given package for event handler classes and registers them on the {@link CdsRuntime}
   *
   * @param packageName the name of the package
   * @return the {@link CdsRuntimeConfigurer}
   * @throws IOException if the classes of the package can't be loaded from the classpath
   */
  CdsRuntimeConfigurer packageScan(String packageName) throws IOException;

  /**
   * Applies all provider configuration modules to the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer providerConfigurations();

  /**
   * Sets the given {@link CdsModelProvider} on the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @param provider the {@link CdsModelProvider}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer provider(CdsModelProvider provider);

  /**
   * Sets the given {@link AuthenticationInfoProvider} on the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @param provider the {@link AuthenticationInfoProvider}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer provider(AuthenticationInfoProvider provider);

  /**
   * Sets the given {@link UserInfoProvider} on the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @param provider the {@link UserInfoProvider}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer provider(UserInfoProvider provider);

  /**
   * Sets the given {@link ParameterInfoProvider} on the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @param provider the {@link ParameterInfoProvider}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer provider(ParameterInfoProvider provider);

  /**
   * Sets the given {@link FeatureTogglesInfoProvider} on the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @param provider the {@link FeatureTogglesInfoProvider}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer provider(FeatureTogglesInfoProvider provider);

  /**
   * Sets the given {@link LocalizedMessageProvider} on the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @param provider the {@link LocalizedMessageProvider}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer provider(LocalizedMessageProvider provider);

  /**
   * Sets the generic {@link CdsProvider} on the {@link CdsRuntime} of this {@link
   * CdsRuntimeConfigurer}
   *
   * @param provider the {@link CdsProvider}
   * @return the {@link CdsRuntimeConfigurer}
   */
  CdsRuntimeConfigurer provider(CdsProvider<?> provider);

  /**
   * Completes the {@link CdsRuntimeConfigurer} to prevent any further modifications of the {@link
   * CdsRuntime}
   *
   * @return the {@link CdsRuntime} in its final state
   */
  CdsRuntime complete();
}
