/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.auditlog;

import com.sap.cds.services.Service;
import java.util.Collections;
import java.util.List;

/** Defines the audit log service API. */
public interface AuditLogService extends Service {

  public static final String DEFAULT_NAME = "AuditLogService$Default";

  /**
   * Writes accesses to sensitive personal data to the audit log.
   *
   * @param accesses a list with data accesses to be logged
   */
  void logDataAccess(List<Access> accesses);

  /**
   * Writes a single data access to sensitive personal data to the audit log.
   *
   * @param access a single data accesses to be logged
   */
  default void logDataAccess(Access access) {
    logDataAccess(Collections.singletonList(access));
  }

  /**
   * Writes modifications to personal data to the audit log.
   *
   * @param modifications a list with all modifications to personal data to be logged
   */
  void logDataModification(List<DataModification> modifications);

  /**
   * Writes a single modification to personal data to the audit log.
   *
   * @param modification a single modification to personal data to be logged
   */
  default void logDataModification(DataModification modification) {
    logDataModification(Collections.singletonList(modification));
  }

  /**
   * Writes multiple configuration changes to the audit log.
   *
   * @param action the performed action
   * @param changes a list with configuration changes
   */
  void logConfigChange(Action action, List<ConfigChange> changes);

  /**
   * Writes a single configuration change to the audit log.
   *
   * @param action the performed action
   * @param change a single configuration change
   */
  default void logConfigChange(Action action, ConfigChange change) {
    logConfigChange(action, Collections.singletonList(change));
  }

  /**
   * Writes a security event in the audit log.
   *
   * @param action the action of the security event
   * @param data the data of the security event
   */
  void logSecurityEvent(String action, String data);
}
