/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.draft;

import com.sap.cds.Result;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.EventName;
import java.util.Map;

/** The {@link EventContext} for the {@link DraftService#EVENT_DRAFT_CREATE} event. */
@EventName(DraftService.EVENT_DRAFT_CREATE)
public interface DraftCreateEventContext extends EventContext {

  /**
   * Creates an {@link EventContext} already overlaid with this interface. The event is set to be
   * {@link DraftService#EVENT_DRAFT_CREATE}
   *
   * @param entityName the name of the entity
   * @return the {@link DraftCreateEventContext}
   */
  static DraftCreateEventContext create(String entityName) {
    return EventContext.create(DraftCreateEventContext.class, entityName);
  }

  @Override
  DraftService getService();

  // IN
  /**
   * @return the {@link CqnInsert} statement
   */
  CqnInsert getCqn();

  /**
   * Sets the {@link CqnInsert} statement
   *
   * @param cqn the {@link CqnInsert} statement
   */
  void setCqn(CqnInsert cqn);

  /**
   * @return {@code true} if the draft is created from an existing active entity (edit scenario),
   *     {@code false} otherwise (new scenario)
   */
  Boolean getHasActiveEntity();

  /**
   * Set to {@code true} if the draft is created from an existing active entity (edit scenario),
   * {@code false} otherwise (new scenario)
   *
   * @param hasActiveEntity {@code true} if the draft is created from an existing active entity
   *     (edit scenario), {@code false} otherwise (new scenario)
   */
  void setHasActiveEntity(boolean hasActiveEntity);

  // OUT
  /**
   * @return the {@link Result} of the {@link DraftService#EVENT_DRAFT_CREATE} event
   */
  Result getResult();

  /**
   * Sets the result of the {@link DraftService#EVENT_DRAFT_CREATE} event
   *
   * @param result a {@link Result} or a list of Maps, which will be transformed into a {@link
   *     Result}
   */
  void setResult(Iterable<? extends Map<String, ?>> result);
}
