/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.messaging;

import com.sap.cds.ql.CdsName;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.EventName;
import java.time.Instant;
import java.util.Map;

/** Describes the content of a CloudEvents structured message */
@EventName("*")
public interface CloudEventMessageEventContext extends EventContext {

  /**
   * Creates a new {@link CloudEventMessageEventContext}
   *
   * @param event the unqualified event name of the modelled event
   * @return the {@link CloudEventMessageEventContext}
   */
  static CloudEventMessageEventContext create(String event) {
    return EventContext.create(event, null).as(CloudEventMessageEventContext.class);
  }

  /**
   * The ID identifies the event.
   *
   * @return the identifier
   */
  String getId();

  /**
   * The ID identifies the event.
   *
   * @param id the identifier
   */
  void setId(String id);

  /**
   * @return The version of the CloudEvents specification which the event uses.
   */
  @CdsName("specversion")
  String getSpecVersion();

  /**
   * @param specVersion The version of the CloudEvents specification which the event uses.
   */
  @CdsName("specversion")
  void setSpecVersion(String specVersion);

  /**
   * The source identifies the instance the event originated in. The instance is an uniquely
   * identifiable instance of a service or application including the tenant if a tenant concept is
   * present.
   *
   * @return the source
   */
  String getSource();

  /**
   * The source identifies the instance the event originated in. The instance is an uniquely
   * identifiable instance of a service or application including the tenant if a tenant concept is
   * present.
   *
   * @param source the source
   */
  void setSource(String source);

  /**
   * @return the type of the event related to the source the event originated in.
   */
  String getType();

  /**
   * @param type the type of the event related to the source the event originated in.
   */
  void setType(String type);

  /**
   * @return the subject of the event in the context of the source the event originated in (e.g. a
   *     certain business object or resource).
   */
  String getSubject();

  /**
   * @param subject the subject of the event in the context of the source the event originated in
   *     (e.g. a certain business object or resource).
   */
  void setSubject(String subject);

  /**
   * @return Content type of the event data.
   * @see #getData()
   */
  @CdsName("datacontenttype")
  String getDataContentType();

  /**
   * @param dataContentType Content type of the event data
   * @see #getData()
   */
  @CdsName("datacontenttype")
  void setDataContentType(String dataContentType);

  /**
   * @return the schema that the event data adheres to.
   * @see #getData()
   */
  @CdsName("dataschema")
  String getDataSchema();

  /**
   * @param dataSchema the schema that the event data adheres to.
   * @see #getData()
   */
  @CdsName("dataschema")
  void setDataSchema(String dataSchema);

  /**
   * @return Timestamp of when the occurrence happened.
   */
  Instant getTime();

  /**
   * @param time Timestamp of when the occurrence happened.
   */
  void setTime(Instant time);

  /**
   * @return The data (payload) of the event.
   */
  Map<String, Object> getData();

  /**
   * @param data The data (payload) of the event.
   */
  void setData(Map<String, Object> data);
}
