/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.messaging;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.EventName;
import java.util.Map;

/** Describes the topic message of the {@link MessagingService} */
@EventName("*")
public interface TopicMessageEventContext extends EventContext {

  /**
   * Creates a new {@link TopicMessageEventContext}
   *
   * @param topic destination topic
   * @return the {@link TopicMessageEventContext}
   */
  static TopicMessageEventContext create(String topic) {
    TopicMessageEventContext context =
        EventContext.create(topic, null).as(TopicMessageEventContext.class);
    context.setIsInbound(false);
    return context;
  }

  /**
   * Returns the inbound flag
   *
   * @return true, if the message has been received from a message broker or false, if it emitted to
   *     a message broker.
   */
  Boolean getIsInbound();

  /**
   * Sets the inbound flag
   *
   * @param inbound true, if the message has been received from a message broker or false, if it
   *     emitted to a message broker.
   */
  void setIsInbound(boolean inbound);

  /**
   * Returns the data map. In case of a cloudevents message the data map corresponds to the map,
   * available under the "data" key.
   *
   * @return the data map
   */
  Map<String, Object> getDataMap();

  /**
   * Sets the data map. In case of a cloudevents message the data map corresponds to the map,
   * available under the "data" key.
   *
   * @param data the data map
   */
  void setDataMap(Map<String, Object> data);

  /**
   * Returns the headers map. In case of a cloudevents message the headers map contains all
   * properties of the message, except for "data".
   *
   * @return the header map
   */
  Map<String, Object> getHeadersMap();

  /**
   * Sets the headers map. In case of a cloudevents message the headers map contains all properties
   * of the message, except for "data".
   *
   * @param headers the header data as a map
   */
  void setHeadersMap(Map<String, Object> headers);

  /**
   * Returns the message ID usually generated by the message broker
   *
   * @return the message ID usually generated by the message broker
   */
  String getMessageId();

  /**
   * Sets the message ID
   *
   * @param id the message ID
   */
  void setMessageId(String id);
}
