/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.mt;

import com.sap.cds.services.Service;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/** Interface for a {@link Service} that provides tenant information. */
public interface TenantProviderService extends Service {

  String DEFAULT_NAME = "TenantProviderService$Default";
  String EVENT_READ_TENANTS = "READ_TENANTS";
  String EVENT_READ_PROVIDER_TENANT = "READ_PROVIDER_TENANT";

  /**
   * @return A list of tenant identifiers that are currently subscribed to this application. In
   *     single tenant scenario the list contains a single {@code null} value representing the
   *     provider tenant.
   */
  default List<String> readTenants() {
    return readTenantsInfo(Set.of(TenantInfo.TENANT)).stream()
        .map(TenantInfo::getTenant)
        .collect(Collectors.toCollection(ArrayList::new));
  }

  /**
   * @param fields A set of fields to retain from the {@link TenantInfo} object.
   * @return A list of {@link TenantInfo} for all tenants that are currently subscribed to this
   *     application. In single tenant scenario the list contains a single {@code TenantInfo} object
   *     with tenant {@code null} representing the provider tenant.
   */
  List<TenantInfo> readTenantsInfo(Set<String> fields);

  /**
   * @return A list of {@link TenantInfo} for all tenants that are currently subscribed to this
   *     application. In single tenant scenario the list contains a single {@code TenantInfo} object
   *     with tenant {@code null} representing the provider tenant.
   */
  List<TenantInfo> readTenantsInfo();

  /**
   * @return The tenant identifier of the tenant that provides this application. In single tenant
   *     scenario the value {@code null} is returned.
   */
  String readProviderTenant();
}
