/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.request;

import java.time.Instant;
import java.util.Locale;
import java.util.Map;

/**
 * Interface to modify the values of the {@link ParameterInfo} used in a {@link
 * com.sap.cds.services.request.RequestContext}.
 */
public interface ModifiableParameterInfo extends ParameterInfo {

  /**
   * Sets the value of the header with the given identifier.
   *
   * @param id the header identifier.
   * @param value the value of the header.
   * @return the same {@link ModifiableParameterInfo} instance
   */
  ModifiableParameterInfo setHeader(String id, String value);

  /**
   * Sets all the header values.
   *
   * @param headers the headers.
   * @return the same {@link ModifiableParameterInfo} instance
   */
  ModifiableParameterInfo setHeaders(Map<String, String> headers);

  /**
   * Sets the value of the query parameter with the given key.
   *
   * @param key the query parameter key.
   * @param value the value of the query parameter.
   * @return the same {@link ModifiableParameterInfo} instance
   */
  ModifiableParameterInfo setQueryParameter(String key, String value);

  /**
   * Sets all the query parameters.
   *
   * @param queryParameters the map of query parameters.
   * @return the same {@link ModifiableParameterInfo} instance
   */
  ModifiableParameterInfo setQueryParameters(Map<String, String> queryParameters);

  /**
   * Sets the {@link Locale} of the request.
   *
   * @param locale the preferred {@link Locale} set by the request.
   * @return the same {@link ModifiableParameterInfo} instance
   */
  ModifiableParameterInfo setLocale(Locale locale);

  /**
   * Sets the "valid-from" time stamp for the request.
   *
   * @param validFrom the {@link Instant} for valid from time stamp.
   * @return the same {@link ModifiableParameterInfo} instance
   */
  ModifiableParameterInfo setValidFrom(Instant validFrom);

  /**
   * Sets the "valid-to" time stamp for the request.
   *
   * @param validTo the {@link Instant} for valid to time stamp.
   * @return the same {@link ModifiableParameterInfo} instance
   */
  ModifiableParameterInfo setValidTo(Instant validTo);

  /**
   * Sets the "valid-at" time stamp for the request. This is a utility method, which sets
   * corresponding values on {@link #setValidFrom(Instant)} and {@link #setValidTo(Instant)}
   *
   * @param validAt the {@link Instant} for valid at time stamp.
   * @return the same {@link ModifiableParameterInfo} instance
   */
  ModifiableParameterInfo setValidAt(Instant validAt);

  /**
   * Sets the correlation ID for the request.
   *
   * @param correlationId the correlation ID.
   * @return the same {@link ModifiableParameterInfo} instance
   */
  ModifiableParameterInfo setCorrelationId(String correlationId);
}
