/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.cds;

import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnUpsert;
import com.sap.cds.services.Service;
import com.sap.cds.services.outbox.OutboxService;
import java.util.Map;

/**
 * The interface defining the asynchronous consumption API of an outboxed {@link CqnService}. All
 * methods of this interface are non-blocking and return immediately since they are submitted to the
 * outbox and executed asynchronously.
 */
public interface AsyncCqnService extends Service {

  /**
   * Returns an outboxed instance of the given {@link CqnService} as a {@link AsyncCqnService}.
   *
   * @param service the {@link CqnService} to be outboxed
   * @param outbox the {@link OutboxService} to be used for outboxing
   * @return an outboxed instance of the given {@link CqnService} as a {@link AsyncCqnService}
   */
  public static AsyncCqnService of(CqnService service, OutboxService outbox) {
    return outbox.outboxed(service, AsyncCqnService.class);
  }

  /**
   * Submits a {@link CqnInsert} statement to the outbox for asynchronous execution.
   *
   * @param insert the {@link CqnInsert} to be executed
   */
  public void run(CqnInsert insert);

  /**
   * Submits a {@link CqnUpsert} statement to the outbox for asynchronous execution.
   *
   * @param upsert the {@link CqnUpsert} to be executed
   */
  public void run(CqnUpsert upsert);

  /**
   * Submits a {@link CqnUpdate} statement with optional positional parameter values to the outbox
   * for asynchronous execution.
   *
   * @param update the {@link CqnUpdate} to be executed
   * @param paramValues the optional positional parameter values
   */
  public void run(CqnUpdate update, Object... paramValues);

  /**
   * Submits a {@link CqnUpdate} statement with named parameter values to the outbox for
   * asynchronous execution.
   *
   * @param update the {@link CqnUpdate} to be executed
   * @param namedValues the named parameter values
   */
  public void run(CqnUpdate update, Map<String, Object> namedValues);

  /**
   * Submits a {@link CqnUpdate} statement as batch with the given named parameter values to the
   * outbox for asynchronous execution.
   *
   * @param update the {@link CqnUpdate} to be executed
   * @param valueSets the named parameter values
   */
  public void run(CqnUpdate update, Iterable<Map<String, Object>> valueSets);

  /**
   * Submits a {@link CqnDelete} statement with optional positional parameter values to the outbox
   * for asynchronous execution.
   *
   * @param delete the {@link CqnDelete} to be executed
   * @param paramValues the optional positional parameter values
   */
  public void run(CqnDelete delete, Object... paramValues);

  /**
   * Submits a {@link CqnDelete} statement with named parameter values to the outbox for
   * asynchronous execution.
   *
   * @param delete the {@link CqnDelete} to be executed
   * @param namedValues the named parameter values
   */
  public void run(CqnDelete delete, Map<String, Object> namedValues);

  /**
   * Submits a {@link CqnDelete} statement as batch with the given named parameter values to the
   * outbox for asynchronous execution.
   *
   * @param delete the {@link CqnDelete} to be executed
   * @param valueSets the named parameter values
   */
  public void run(CqnDelete delete, Iterable<Map<String, Object>> valueSets);
}
