/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.cds;

import com.sap.cds.Result;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.EventName;
import java.util.Map;

/** The specialized {@link EventContext} to be used for {@link CqnService#EVENT_READ} */
@EventName(CqnService.EVENT_READ)
public interface CdsReadEventContext extends EventContext {

  /**
   * Creates an {@link EventContext} already overlaid with this interface. The event is set to be
   * {@link CqnService#EVENT_READ}
   *
   * @param entityName the name of the entity
   * @return the {@link CdsReadEventContext}
   */
  static CdsReadEventContext create(String entityName) {
    return EventContext.create(CdsReadEventContext.class, entityName);
  }

  @Override
  CqnService getService();

  // IN
  /**
   * @return the {@link CqnSelect} statement
   */
  CqnSelect getCqn();

  /**
   * Sets the {@link CqnSelect} statement
   *
   * @param cqn the {@link CqnSelect} statement
   */
  void setCqn(CqnSelect cqn);

  /**
   * The CQN named values, for parameterized {@link CqnSelect} statements
   *
   * @return the CQN named values
   */
  Map<String, Object> getCqnNamedValues();

  /**
   * Sets the CQN named values, for parameterized {@link CqnSelect} statements
   *
   * @param namedValues the CQN named values
   */
  void setCqnNamedValues(Map<String, Object> namedValues);

  // OUT
  /**
   * @return the {@link Result} of the {@link CqnService#EVENT_READ} event
   */
  Result getResult();

  /**
   * Sets the result of the {@link CqnService#EVENT_READ} event
   *
   * @param result a {@link Result} or a list of Maps, which will be transformed into a {@link
   *     Result}
   */
  void setResult(Iterable<? extends Map<String, ?>> result);
}
