/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.changeset;

import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.services.CoreFactory;
import com.sap.cds.services.Service;
import com.sap.cds.services.handler.Handler;

/**
 * A {@link ChangeSetContext} defines the transactional boundaries around one or multiple {@link
 * Service#emit(com.sap.cds.services.EventContext)} calls. It can be used by {@link Handler} or
 * other participants to ensure that changes, which were done as part of a ChangeSetContext are
 * canceled (rolled back). <br>
 * In addition, it provides the ability to perform actions at the end of a ChangeSet by registering
 * a {@link ChangeSetListener}
 */
public interface ChangeSetContext {

  /**
   * @return {@code true}, if a {@link ChangeSetContext} is currently active
   */
  static boolean isActive() {
    return CoreFactory.INSTANCE.createChangeSetContextAccessor().isActive();
  }

  /**
   * Gives access to the currently active {@link ChangeSetContext}. If no {@link ChangeSetContext}
   * is currently opened this method will return <code>null</code>
   *
   * @return the currently active {@link ChangeSetContext} or <code>null</code>.
   */
  static ChangeSetContext getCurrent() {
    return CoreFactory.INSTANCE.createChangeSetContextAccessor().getCurrent();
  }

  /**
   * @return a numeric identifier of the {@link ChangeSetContext}, used for logging
   */
  int getId();

  /**
   * Marks the {@link ChangeSetContext} to be cancelled. When the ChangeSet is finished,
   * transactions will be rolled back.
   */
  void markForCancel();

  /**
   * @return true, if {@link #markForCancel()} was called at least once, false otherwise
   */
  boolean isMarkedForCancel();

  /**
   * Marks the {@link ChangeSetContext} as transactional.
   *
   * <p>Transactional {@link ChangeSetContext}s ensure to allocate transactions and corresponding
   * resources (e.g. SQL connections), upon first interaction with a transactional data sources.
   * Those resources are released only after closing the {@link ChangeSetContext}. A transactional
   * data source might mark the {@link ChangeSetContext} as transactional, if it requires a
   * transaction (e.g. when executing a {@link CqnInsert}, {@link CqnUpdate}, {@link CqnDelete} or
   * {@link CqnSelect} with locks).
   *
   * <p>{@link ChangeSetContext} that are not marked transactional allow transactional data sources
   * to free resources earlier, e.g. after each interaction. For example in case of JDBC data
   * sources, SQL connections can be returned to the connection pool after executing a {@link
   * CqnSelect} without locks. In that case they are not required to be kept for the entire {@link
   * ChangeSetContext} scope.
   *
   * <p>Once a {@link ChangeSetContext} is marked as transactional it can't be unmarked again. If
   * the {@link ChangeSetContext} has been marked as transactional already this method doesn't have
   * any effects.
   */
  void markTransactional();

  /**
   * @return {@code true}, if this {@link ChangeSetContext} was marked transactional (see {@link
   *     ChangeSetContext#markTransactional()}).
   */
  boolean isMarkedTransactional();

  /**
   * Registers a {@link ChangeSetListener} on the {@link ChangeSetContext}, to be notified once the
   * {@link ChangeSetContext} is closed successfully or cancelled.
   *
   * @param listener the {@link ChangeSetListener}
   */
  void register(ChangeSetListener listener);
}
