/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.request;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.CoreFactory;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.authentication.AuthenticationInfo;
import com.sap.cds.services.messages.Messages;
import com.sap.cds.services.runtime.CdsRuntime;

/** Interface to provide access to request-specific information. */
public interface RequestContext {

  /**
   * @return true, if a {@link RequestContext} is currently active
   */
  static boolean isActive() {
    return CoreFactory.INSTANCE.createRequestContextAccessor().isActive();
  }

  /**
   * Gives access to either the current {@link RequestContext}, or if this does not exist to a
   * helper {@link RequestContext} that itself ensures that there is always a {@link RequestContext}
   * opened. This method will throw a {@link NullPointerException}, if the passed {@link CdsRuntime}
   * was null, but guarantees that it will never return <code>null</code>
   *
   * @param runtime A reference to the underlying {@link CdsRuntime} instance.
   * @return the current {@link RequestContext} or a helper {@link RequestContext}
   */
  static RequestContext getCurrent(CdsRuntime runtime) {
    return CoreFactory.INSTANCE.createRequestContextAccessor().getCurrent(runtime);
  }

  /**
   * Returns the tenant-specific {@link CdsModel}. The tenant is determined based on the
   * request-context and the tenant information available in {@link UserInfo}. If no tenant is
   * specified, or if no tenant-specific model exists, the base model is returned.
   *
   * @return the (tenant-specific) {@link CdsModel}
   */
  CdsModel getModel();

  /**
   * Returns the {@link ServiceCatalog}, populated with all available {@link Service} instances.
   *
   * @return the {@link ServiceCatalog}
   */
  ServiceCatalog getServiceCatalog();

  /**
   * @return the {@link ParameterInfo} of this request
   */
  ParameterInfo getParameterInfo();

  /**
   * @return the {@link UserInfo} of this request
   */
  UserInfo getUserInfo();

  /**
   * @return the {@link AuthenticationInfo} of this request
   */
  AuthenticationInfo getAuthenticationInfo();

  /**
   * @return the {@link FeatureTogglesInfo} for this request
   */
  FeatureTogglesInfo getFeatureTogglesInfo();

  /**
   * @return the {@link Messages} attached to this request
   */
  Messages getMessages();
}
