/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl;

import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.sap.cds.Struct;
import com.sap.cds.feature.util.ClassMethods;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.EventName;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.impl.request.RequestContextImpl;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.impl.utils.NameOnlyCdsEntity;
import com.sap.cds.services.messages.Messages;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.ResultUtils;
import com.sap.cds.services.utils.StringUtils;


public class EventContextImpl implements EventContextSPI {

	private final static ClassMethods eventContextImplMethods = ClassMethods.create(EventContextImpl.class);

	private final Map<String, Object> map = new HashMap<>();

	private Service service;
	private final String event;
	private final String entityName;
	private CdsEntity target;

	private boolean completed = false;

	public EventContextImpl(String event, String entityName) {
		this.event = event;
		this.entityName = StringUtils.isEmpty(entityName) ? null : entityName;
	}

	@Override
	public CdsModel getModel() {
		return getRequestContext().getModel();
	}

	@Override
	public ServiceCatalog getServiceCatalog() {
		return getRequestContext().getServiceCatalog();
	}

	@Override
	public ParameterInfo getParameterInfo() {
		return getRequestContext().getParameterInfo();
	}

	@Override
	public UserInfo getUserInfo() {
		return getRequestContext().getUserInfo();
	}

	@Override
	public Messages getMessages() {
		return getRequestContext().getMessages();
	}

	@Override
	public CdsRuntime getCdsRuntime() {
		ServiceSPI serviceSPI = CdsServiceUtils.getServiceSPI(service);
		return serviceSPI != null ? serviceSPI.getCdsRuntime() : null;
	}

	// This throws an NPE, as long as the EventContext was not emitted on a service,
	// or more precise as long as setService() was not called
	private RequestContext getRequestContext() {
		return RequestContext.getCurrent(getCdsRuntime());
	}

	@Override
	public ChangeSetContext getChangeSetContext() {
		return ChangeSetContext.getCurrent();
	}

	@Override
	public Service getService() {
		return service;
	}

	@Override
	public void setService(Service service) {
		this.service = service;
	}

	@Override
	public String getEvent() {
		return event;
	}

	@Override
	public CdsEntity getTarget() {
		if(entityName == null) {
			return null;
		}

		if(target == null) {
			RequestContext context = RequestContextImpl.getCurrentOrNull(getCdsRuntime());
			if(context != null && context.getModel() != null) {
				target = CdsModelUtils.getEntityOrThrow(context.getModel(), entityName);
			}
		}
		return target != null ? target : new NameOnlyCdsEntity(entityName);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T extends EventContext> T as(Class<T> clazz) {
		// if this object is already of type T, simply return it
		if(clazz.isAssignableFrom(this.getClass())) {
			return (T) this;
		}

		// make sure the EventContext to be casted matches the event type
		EventName expectedEventName = clazz.getAnnotation(EventName.class);
		if (expectedEventName == null || StringUtils.isEmpty(expectedEventName.value())) {
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_CONTEXT_MISSING_ANNOTATION, clazz.getName());

		} else if ( !expectedEventName.value().equals("*") && !event.equals(expectedEventName.value())) {
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_CONTEXT_EVENT_MISMATCH,
					clazz.getName(), expectedEventName.value(), event);
		}

		// interface laid over data map
		T mapAccessor = Struct.access(map).as(clazz);

		// proxy combining access to EventContext methods and methods of the typed map accessor
		return (T) Proxy.newProxyInstance(clazz.getClassLoader(), new Class[] { clazz, EventContextSPI.class },
				(proxy, method, methodArgs) -> {
					// if the method exists on the generic event context, invoke directly on this object
					Method originalMethod = eventContextImplMethods.lookupMethod(method);
					if(originalMethod != null) {
						// prevent creating further proxies
						if(method.getName().equals("as") && methodArgs.length == 1 && ((Class<?>) methodArgs[0]).isAssignableFrom(clazz)) {
							return proxy;
						}

						return originalMethod.invoke(this, methodArgs);
					}

					// TODO generalize this concept
					if(method.getName().equals("setResult")) {
						Type[] types = method.getGenericParameterTypes();
						if(types.length == 1 && types[0].getTypeName().equals("java.lang.Iterable<? extends java.util.Map<java.lang.String, ?>>")) {
							methodArgs[0] = ResultUtils.convert((Iterable<? extends Map<String, ?>>) methodArgs[0]);
						}
					}

					// method only exists on the interface
					Object obj = method.invoke(mapAccessor, methodArgs);

					// TODO generalize this concept
					// automatic completion, when result is set
					if(method.getName().equals("setResult")) {
						setCompleted();
					}

					return obj;
				});
	}

	@Override
	public Set<String> keySet() {
		return Collections.unmodifiableSet(map.keySet());
	}

	@Override
	public Object get(String key) {
		return map.get(key);
	}

	@Override
	public void put(String key, Object value) {
		map.put(key, value);
	}

	@Override
	public void setCompleted() {
		this.completed = true;
	}

	@Override
	public boolean isCompleted() {
		return completed == true;
	}

}
