/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.authorization;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.model.Restriction;

public class RestrictionLookup {

	private final Map<String, Restriction> cachedRestrictions = new ConcurrentHashMap<>();

	private static final Restriction undefined = new Restriction();

	public Restriction retrieveServiceRestriction(CdsModel model, String serviceName) {
		return lookupRestriction(serviceName, (name) -> com.sap.cds.services.utils.model.CdsModelUtils.getRestrictionOrNull( CdsModelUtils.getServiceOrThrow(model, name) ));
	}

	public Restriction retrieveEntityRestriction(CdsModel model, String entityName) {
		return lookupRestriction(entityName, (name) -> com.sap.cds.services.utils.model.CdsModelUtils.getRestrictionOrNull( CdsModelUtils.getEntityOrThrow(model, name) ));
	}

	public Restriction lookupFunctionRestriction(CdsModel model, String entityName, String functionName) {
		String key = (entityName != null ? (entityName + "." + functionName) : functionName);
		return lookupRestriction(key, (name) -> com.sap.cds.services.utils.model.CdsModelUtils.getRestrictionOrNull( CdsModelUtils.getFunction(model, entityName, functionName)
				.orElseThrow(() -> new ErrorStatusException(CdsErrorStatuses.FUNCTION_NOT_FOUND, functionName))));
	}

	public Restriction lookupActionRestriction(CdsModel model, String entityName, String actionName) {
		String key = (entityName != null ? (entityName + "." + actionName) : actionName);
		return lookupRestriction(key, (name) -> com.sap.cds.services.utils.model.CdsModelUtils.getRestrictionOrNull( CdsModelUtils.getAction(model, entityName, actionName)
				.orElseThrow(() -> new ErrorStatusException(CdsErrorStatuses.ACTION_NOT_FOUND, actionName))));
	}

	private Restriction lookupRestriction(String name, Function<String, Restriction> resolver) {

		Restriction cachedRestriction = cachedRestrictions.get(name);
		if (cachedRestriction == null) {
			cachedRestriction = resolver.apply(name);
			// ConcurrentHashMap does not support null values
			cachedRestrictions.put(name, cachedRestriction == null ? undefined : cachedRestriction);
		}

		// transform undefined back into null
		return cachedRestriction == undefined ? null : cachedRestriction;
	}

}
