/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.authorization;

import static com.sap.cds.services.utils.model.Privilege.PredefinedGrant.ALL;
import static com.sap.cds.services.utils.model.Privilege.PredefinedGrant.WRITE;
import static com.sap.cds.services.utils.model.Privilege.PredefinedRole.ANY_USER;
import static com.sap.cds.services.utils.model.Privilege.PredefinedRole.AUTHENTICATED_USER;
import static com.sap.cds.services.utils.model.Privilege.PredefinedRole.SYSTEM_USER;

import java.util.Objects;
import java.util.stream.Stream;

import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.utils.model.Privilege;
import com.sap.cds.services.utils.model.Restriction;

public class RestrictionUtils {

	private RestrictionUtils() {
		// hidden
	}

	/**
	 * Returns {@code true} if and only if the given user passes the given {@link Restriction} for the given event, i.e.
	 * at least one of the privileges of the restriction is matched.
	 *
	 * @param restriction	The {@link Restriction} to be tested
	 * @param user	The user to be tested
	 * @param event	The event to be tested
	 * @return	{@code true} if and only if the given user passes the given restriction for the event
	 */
	public static boolean passesRestriction(Restriction restriction, UserInfo user, String event) {
		// a single privilege which is passed is sufficient for positive signal
		return passingPrivilegesOfRestriction(restriction, user, event).findAny().isPresent();
	}


	/**
	 * Returns all {@link Privilege}s of the passed {@link Restriction} which are passed by the current {@link UserInfo} with given event.
	 *
	 * @param restriction	The {@link Restriction} to be analyzed
	 * @param user	The user to be tested
	 * @param event	The event to be tested
	 * @return	The passed privileges of empty Stream if not existing
	 */
	public static Stream<Privilege> passingPrivilegesOfRestriction(Restriction restriction, UserInfo user, String event) {
		Objects.requireNonNull(restriction, "Restriction required");
		Objects.requireNonNull(user, "UserInfo required");
		Objects.requireNonNull(event, "event required");

		return restriction.privileges().filter(privilege -> hasPrivilege(privilege, user, event));
	}

	/**
	 * Returns {@code true} if and only if the given user matches the given privilege for the given event, i.e.
	 * the privilege grants the event and the user provides an matching role.
	 *
	 * @param privilege	The privilege to be tested
	 * @param user	The user to be tested
	 * @param event	The event to be tested
	 * @return	{@code true} if and only if the given user matches the given privilege
	 */
	public static boolean hasPrivilege(Privilege privilege, UserInfo user, String event) {

		if(privilege == null || user == null || event == null) {
			return false;
		}
		else if (user.isPrivileged()) {
			return true;
		}

		// first check grants
		boolean matchesGrant = false;
		for (String grant : privilege.getGrants()) {
			if (ALL.is(grant) ||  // '*' means all events match the privilege
					Privilege.is(event, grant) || 	// event exactly matches the grant
					(WRITE.is(grant) && CdsModelUtils.isWriteEvent(event)) ||  // grant contains 'WRITE' and event has write semantic
					CdsModelUtils.eventIsGranted(event, grant)	) // some events are mapped to others (e.g. draft events)
			{
				matchesGrant = true;
				break;
			}
		}

		if (!matchesGrant) {
			return false;
		}
		assert matchesGrant == true;

		// then check roles. One match is sufficient if there is at least one role.
		if (privilege.getRoles().isEmpty()) {
			return true; // no roles required -> public access
		}

		boolean matchesRole = false;
		for (String role : privilege.getRoles()) {
			if (ANY_USER.is(role)) { // no authentication (neither named nor system ) is required here!
				matchesRole = true;

			} else if (AUTHENTICATED_USER.is(role)) {
				matchesRole = user.isAuthenticated();

			} else if (SYSTEM_USER.is(role)) {
				matchesRole = user.isSystemUser();

			} else if (user.hasRole(role)) {
				matchesRole = user.isAuthenticated() || user.isSystemUser();
			}
			if (matchesRole) {
				break;
			}
		}

		return matchesRole;
	}
}
