/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.DELETABLE;
import static com.sap.cds.services.utils.model.CdsAnnotations.INSERTABLE;
import static com.sap.cds.services.utils.model.CdsAnnotations.INSERTONLY;
import static com.sap.cds.services.utils.model.CdsAnnotations.READONLY;
import static com.sap.cds.services.utils.model.CdsAnnotations.UPDATABLE;

import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.services.utils.model.CdsModelUtils;

@ServiceName(value = "*", type = ApplicationService.class)
public class CapabilitiesHandler implements EventHandler {

	// default handlers
	@Before(event = CdsService.EVENT_READ)
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	public void checkCapabilityRead(EventContext context) {
		if(!getCapabilities(context).isReadable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_READABLE, context.getTarget().getQualifiedName());
		}
	}

	@Before(event = { CdsService.EVENT_CREATE, CdsService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	public void checkCapabilityCreate(EventContext context) {
		if(!getCapabilities(context).isInsertable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_INSERTABLE, context.getTarget().getQualifiedName());
		}
	}

	@Before(event = { CdsService.EVENT_UPDATE, CdsService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	public void checkCapabilityUpdate(EventContext context) {
		if(!getCapabilities(context).isUpdatable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_UPDATABLE, context.getTarget().getQualifiedName());
		}
	}

	@Before(event = CdsService.EVENT_DELETE)
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	public void checkCapabilityDelete(EventContext context) {
		if(!getCapabilities(context).isDeletable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_DELETABLE, context.getTarget().getQualifiedName());
		}
	}

	/**
	 * Returns the {@link Capabilities} for a {@link CdsEntity}.
	 * The {@link Capabilities} describe CRUD restrictions on the entity.
	 *
	 * Annotations {@literal @readonly} and {@literal @insertonly} take precedence before
	 * {@literal @Capabilities.Insertable}, {@literal @Capabilities.Updatable}, {@literal @Capabilities.Deletable}
	 *
	 * @param context the {@link EventContext} whose target {@link CdsEntity} should be checked
	 * @return the {@link Capabilities}
	 */
	public static Capabilities getCapabilities(EventContext context) {
		CdsEntity entity = context.getTarget();
		boolean autoexposed = CdsAnnotations.AUTOEXPOSED.getOrDefault(entity);
		boolean autoexpose = CdsAnnotations.AUTOEXPOSE.getOrDefault(entity);
		// Auto exposed composition entities reject all direct CRUD requests
		// Explicitly auto-exposed entities reject all direct C_UD requests (READ is allowed, required by Fiori Value Helps)
		// Exception: In draft scenario all CRUD requests are required by Fiori
		CqnStatement cqn = (CqnStatement) context.get("cqn");
		if(isDirectlyReferenced(cqn, context.getModel()) && autoexposed && !DraftUtils.isDraftEnabled(entity)) {
			if(autoexpose) {
				return new Capabilities(true, false, false, false);
			}
			return new Capabilities(false, false, false, false);
		}

		// readonly and insertonly take precedence
		boolean readonly = READONLY.getOrDefault(entity);
		if(readonly) {
			return new Capabilities(true, false, false, false);
		}

		boolean insertOnly = INSERTONLY.getOrDefault(entity);
		if(insertOnly) {
			return new Capabilities(false, true, false, false);
		}

		// capabilities default to true
		boolean insertable = defaultToTrue(INSERTABLE.getOrDefault(entity));
		boolean updatable = defaultToTrue(UPDATABLE.getOrDefault(entity));
		boolean deletable = defaultToTrue(DELETABLE.getOrDefault(entity));

		return new Capabilities(true, insertable, updatable, deletable);
	}

	/**
	 * This method ensures, that the capability is only set to false, if "false" was provided in the annotation.
	 * True or any other string are treated as true.
	 * @param value
	 * @return
	 */
	private static boolean defaultToTrue(Object value) {
		if(value instanceof Boolean) {
			return (boolean) value;
		} else if (value instanceof String) {
			return !((String) value).equalsIgnoreCase("false");
		}

		return true;
	}

	private static boolean isDirectlyReferenced(CqnStatement cqn, CdsModel model) {
		if (cqn.isSelect() && cqn.asSelect().from().isSelect()) {
			return isDirectlyReferenced(cqn.asSelect().from().asSelect(), model);
		}
		AnalysisResult entityPath = CdsModelUtils.getEntityPath(cqn.ref(), model);
		CdsEntity rootEntity = entityPath.iterator().next().entity();
		// if target and root are the same the entity is directly referenced
		return rootEntity.getQualifiedName().equals(entityPath.targetEntity().getQualifiedName());
	}

	public static class Capabilities {

		private boolean readable;
		private boolean insertable;
		private boolean updatable;
		private boolean deletable;

		private Capabilities(boolean readable, boolean insertable, boolean updatable, boolean deletable) {
			this.readable = readable;
			this.insertable = insertable;
			this.updatable = updatable;
			this.deletable = deletable;
		}

		public boolean isReadable() {
			return readable;
		}

		public boolean isInsertable() {
			return insertable;
		}

		public boolean isUpdatable() {
			return updatable;
		}

		public boolean isDeletable() {
			return deletable;
		}

	}

}
