/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.ASSERT_RANGE;

import java.util.Map;
import java.util.stream.Collectors;

import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEnumType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = ApplicationService.class)
public class EnumAssertionHandler implements EventHandler {

	@Before(event = { CdsService.EVENT_CREATE, CdsService.EVENT_UPDATE, CdsService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		CdsModelUtils.visitDeep(context.getTarget(), CdsServiceUtils.getEntities(context), (entity, data, parent) -> {
			entity.elements().forEach(element -> {

				// Skip the current element ...
				// ... if the element is not of type enum
				if (!element.getType().isEnum()) {
					return;
				}
				// ... if no @assert.range annotation exists
				if(Boolean.FALSE.equals(ASSERT_RANGE.getOrDefault(element))) {
					return;
				}

				for (Map<String, Object> elementMap : data) {
					Object elementValue = elementMap.get(element.getName());
					if (elementValue != null) {
						 ensureValidEnum(elementValue, element);
					}
				}
			});
		});
	}

	private void ensureValidEnum(Object elementValue, CdsElement element) {
		CdsEnumType<?> enumElement = element.getType().as(CdsEnumType.class);
		if (!enumElement.hasValue(elementValue)) {
			String validEnumValues = enumElement.enumerals().values().stream().map(e -> e.value() + " (" + e.name() + ")").collect(Collectors.joining(", "));
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_ENUM, elementValue, element.getName(), validEnumValues);
		}
	}

}
