/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.ASSERT_FORMAT;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.TenantAwareCache;

@ServiceName(value = "*", type = ApplicationService.class)
public class FormatAssertionHandler implements EventHandler {

	// cache the regex pattern due to Performance of pattern compilation
	private final TenantAwareCache<Map<String, Pattern>, CdsModel> patternCache;

	public FormatAssertionHandler(CdsRuntime runtime) {
		this.patternCache = TenantAwareCache.create(() -> new HashMap<String, Pattern>(), runtime);
	}

	@Before(event = { CdsService.EVENT_CREATE, CdsService.EVENT_UPDATE, CdsService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		CdsModelUtils.visitDeep(context.getTarget(), CdsServiceUtils.getEntities(context), (entity, data, parent) -> {

			entity.elements().forEach(element -> {

				// Skip the current element ...
				// ... if the element is not of type String
				if (!(element.getType() instanceof CdsSimpleType) || element.getType().as(CdsSimpleType.class).getType() != CdsBaseType.STRING) {
					return;
				}
				// ... if no @assert.format annotation exists or is not of type String
				Object regex = ASSERT_FORMAT.getOrValue(element, null);
				if (regex == null || !(regex instanceof String)) {
					return;
				}
				for (Map<String, Object> elementMap : data) {
					Object elementValue = elementMap.get(element.getName());
					if (elementValue != null) {
						// Throw an exception if the annotation value is not a valid pattern
						Pattern pattern = getPattern((String) regex, element.getName(), entity.getQualifiedName());
						// Throw an exception if the element value does not match the pattern
						ensureValidFormat(elementValue.toString(), pattern, element.getName());
					}
				}
			});
		});
	}

	private Pattern getPattern(String regex, String elementName, String entityName) {
		Pattern pattern = this.patternCache.findOrCreate().get(regex);
 		if (pattern == null) {
			// Try to create cached Pattern
			try {
				pattern = Pattern.compile(regex);
			} catch (PatternSyntaxException e) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_ANNOTATION, ASSERT_FORMAT.toString(), elementName, entityName, e);
			}
			this.patternCache.findOrCreate().put(regex, pattern);
		}
		return pattern;
	}

	private void ensureValidFormat(String elementConvertedValue, Pattern pattern, String elementName) {
		Matcher matcher = pattern.matcher(elementConvertedValue);
		if (!matcher.matches()){
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_STRING_VALUE, elementConvertedValue, elementName);
		}

	}

}
