/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.ASSERT_RANGE;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.sap.cds.ql.CdsDataException;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.util.CdsTypeUtils;

@ServiceName(value = "*", type = ApplicationService.class)
public class RangeAssertionHandler implements EventHandler {

	@SuppressWarnings("unchecked")
	@Before(event = { CdsService.EVENT_CREATE, CdsService.EVENT_UPDATE, CdsService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		CdsModelUtils.visitDeep(context.getTarget(), CdsServiceUtils.getEntities(context), (entity, data, parent) -> {

			entity.elements().forEach(element -> {

				// Skip the current element ...
				// ... if no @assert.range annotation exists
				Object rangeValues = ASSERT_RANGE.getListOrValue(element, null);
				if (rangeValues == null) return;
				// ... if the annotation value is not of type List with two values
				if(!(rangeValues instanceof List)) return;
				if(((List<?>) rangeValues).size() != 2) return;
				// ... if the element is not an allowed type
				if(!isAllowedType(element.getType())) return;

				List<Comparable<Object>> rangeConvertedValues = getConvertedRange((List<Object>) rangeValues, element, entity.getName());

				for (Map<String, Object> elementMap : data) {
					Object elementValue =  elementMap.get(element.getName());
					if (elementValue != null) {
						ensureValidRange((Comparable<Object>)elementValue, rangeConvertedValues);
					}
				}
			});
		});
	}

	private void ensureValidRange(Comparable<Object> elementValue, List<Comparable<Object>> rangeValues) {
		Comparable<Object> minValue = rangeValues.get(0);
		Comparable<Object> maxValue = rangeValues.get(1);

		// Throw an exception if the value is not within the range
		if(elementValue.compareTo(maxValue) > 0 || elementValue.compareTo(minValue) < 0 ){
			throw new ErrorStatusException(CdsErrorStatuses.VALUE_OUT_OF_RANGE, elementValue, minValue, maxValue);
		}
	}

	@SuppressWarnings("unchecked")
	private List<Comparable<Object>> getConvertedRange(List<Object> rangeValues, CdsElement element, String entityName) {
		try {
			return rangeValues.stream()
			.map(oValue -> (Comparable<Object>) CdsTypeUtils.parse(element.getType().as(CdsSimpleType.class).getType(), oValue.toString()))
			.collect(Collectors.toList());
		} catch (CdsDataException e) {
			// Throw an exception if the annotation values do not match the element
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_ANNOTATION, ASSERT_RANGE.toString(), element.getName(), entityName, e);
		}
	}

	@SuppressWarnings("deprecation")
	private boolean isAllowedType(CdsType elementType){

		if (!(elementType instanceof CdsSimpleType)) return false;

		// Allowed Types for @assert.range
		switch (elementType.as(CdsSimpleType.class).getType()) {
			case INTEGER:
			case INTEGER64:
			case DECIMAL:
			case DECIMAL_FLOAT:
			case DOUBLE:
			case DATE:
			case TIME:
			case DATETIME:
			case TIMESTAMP:
				return true;
			default:
				return false;
		}

	}
}
