/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;

@ServiceName(value = "*", type = ApplicationService.class)
public class ReadOnlyHandler implements EventHandler {

	@Before(event = { CdsService.EVENT_CREATE, CdsService.EVENT_UPDATE, CdsService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.FILTER_FIELDS)
	public void cleanReadOnlyFields(EventContext context) {
		String event = context.getEvent();

		List<Map<String, Object>> entries = CdsServiceUtils.getEntities(context);

		// traverse the entity deep
		CdsModelUtils.visitDeep(context.getTarget(), entries, (entity, data, parent) -> {
			// find all read only elements
			Set<String> readOnlyElementNames = entity.elements().filter(element -> isReadOnly(element, event)).map(e -> e.getName()).collect(Collectors.toSet());
			if(readOnlyElementNames.isEmpty()) {
				return;
			}

			// remove read only elements from the map
			for(Map<String, Object> map : data) {
				for(String e : readOnlyElementNames) {
					map.remove(e);
				}
			}
		});
	}

	/**
	 * Returns true, if a {@link CdsElement} is read only
	 * @param element the {@link CdsElement}
	 * @param event the event currently being processed
	 * @return true, if a {@link CdsElement} is read only
	 */
	public boolean isReadOnly(CdsElement element, String event) {
		if (element == null) {
			return false;
		}

		boolean onUpdate = CdsAnnotations.ON_UPDATE.getOrDefault(element) != null;
		boolean onInsert = CdsAnnotations.ON_INSERT.getOrDefault(element) != null;

		// TODO cds.on.insert should actually be readonly during UPSERTs -> however this would remove createdAt timestamps for example during OData V4 PUTs
		if(onUpdate || (onInsert && !CdsService.EVENT_UPSERT.equals(event))) {
			return true;
		}

		// Check FieldControl ReadOnly annotations
		boolean readOnly = CdsAnnotations.READONLY.getOrDefault(element);
		boolean fieldControlReadOnly = CdsAnnotations.FIELD_CONTROL_READONLY.getOrDefault(element);
		Map<?, ?> commonFieldControl = CdsAnnotations.COMMON_FIELDCONTROL.getOrDefault(element);
		if(readOnly || fieldControlReadOnly || (commonFieldControl != null && "ReadOnly".equals(commonFieldControl.get("#")))) {
			return true;
		}

		// Core.Computed is read only
		// onUpdate and onInsert are also Core.Computed, but already handled above
		boolean coreComputed = CdsAnnotations.CORE_COMPUTED.getOrDefault(element);
		if(coreComputed && !onUpdate && !onInsert && !element.isKey()) {
			return true;
		}

		// TODO what about UPSERT?
		// Core.Immutable is read only during update
		boolean coreImmutable = CdsAnnotations.CORE_IMMUTABLE.getOrDefault(element);
		if(coreImmutable && CdsService.EVENT_UPDATE.equals(event)) {
			return true;
		}

		// virtual is read only
		return element.isVirtual();
	}

}
