/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;

import com.sap.cds.feature.config.Properties;
import com.sap.cds.impl.parser.builder.ExpressionBuilder;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.Value;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnModifier;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.impl.utils.CdsModelUtils;

/**
 * A {@link CqnModifier} for the all parts of draft statements except for reference, aggregations, orderby and columns.
 */
public class DraftModifier implements CqnModifier {

	private boolean activeEntityOnly;
	private boolean forDraftEntity;
	private CdsEntity entity;
	private final boolean adaptIsActiveEntity;

	public DraftModifier(CdsEntity entity, boolean adaptIsActiveEntity) {
		this(entity, false, adaptIsActiveEntity);
	}

	public DraftModifier(CdsEntity entity, boolean activeEntityOnly, boolean adaptIsActiveEntity) {
		this.entity = entity;
		this.forDraftEntity = entity.getQualifiedName().endsWith(CqnAdapter.DRAFT_SUFFIX);
		this.activeEntityOnly = activeEntityOnly;
		this.adaptIsActiveEntity = adaptIsActiveEntity;
	}

	@Override
	public Value<?> ref(ElementRef<?> ref) {
		return adaptElement(ref, entity, forDraftEntity, activeEntityOnly, adaptIsActiveEntity);
	}

	public static Value<?> adaptElement(CqnElementRef ref, CdsEntity entity, boolean forDraftEntity, boolean activeEntityOnly, boolean adaptIsActiveEntity) {
		List<Segment> segments = new ArrayList<>(ref.segments());
		if (referencesDraftField(Drafts.IS_ACTIVE_ENTITY, segments, entity, forDraftEntity) && adaptIsActiveEntity) {
			return CQL.literal(true);
		} else if (!activeEntityOnly) {
			if (DraftAdministrativeData.IN_PROCESS_BY_USER.equals(ref.targetSegment().id())) {
				return createTimeoutExpression(ref);
			}
		}
		return CQL.get(segments);
	}

	public static void insertFieldBefore(String id, List<Segment> segments, Segment toInsert) {
		for (int i = 0; i < segments.size(); ++i) {
			if (segments.get(i).id().equals(id)) {
				segments.add(i, toInsert);
				return;
			}
		}
	}

	public static Instant getTimeoutThreshold() {
		// TODO should we be able to get the NOW time stamp from the context?
		return Instant.now().minus(Properties.getCds().getDrafts().getCancellationTimeout()).truncatedTo(ChronoUnit.MILLIS);
	}

	public static ElementRef<?> changeLastSegment(CqnElementRef ref, String id) {
		List<Segment> result = new ArrayList<>(ref.segments());
		result.set(result.size() - 1, CQL.refSegment(id));
		return CQL.get(result);
	}

	public static boolean referencesDraftField(String draftColumn, List<? extends Segment> segments, CdsEntity entity, boolean forDraftEntity) {
		if (segments.stream().anyMatch(s -> s.id().equalsIgnoreCase(draftColumn))) {
			for (Segment segment: segments) {
				String id = segment.id();
				if (draftColumn.equalsIgnoreCase(id)) {
					return !forDraftEntity;
				}
				if (CdsModelUtils.isAssociationToParentOrChild(id, entity)) {
					entity = entity.getTargetOf(id);
				} else if (ReferenceModifier.SIBLING.equals(id) || ReferenceModifier.SIBLING_UNSECURED.equals(id)) {
					forDraftEntity = !forDraftEntity;
				} else if (entity.findAssociation(id).isPresent()) {
					entity = entity.getTargetOf(id);
					forDraftEntity = id.endsWith(CqnAdapter.DRAFT_SUFFIX);
				}
			}
		}

		return false;
	}

	public static Value<?> createTimeoutExpression(CqnElementRef ref) {
		ElementRef<?> refLastChangeDateTime = changeLastSegment(ref, DraftAdministrativeData.LAST_CHANGE_DATE_TIME);
		ElementRef<?> refDraftUuid = changeLastSegment(ref, DraftAdministrativeData.DRAFT_UUID);
		return (Value<?>) ExpressionBuilder.xpr()
				.plain("CASE")
				.plain("WHEN")
				.ref(refDraftUuid)
				.plain("IS NULL")
				.plain("THEN")
				.plain("null")
				.plain("WHEN")
				.ref(refLastChangeDateTime)
				.plain(">")
				.token(CQL.literal(getTimeoutThreshold()))
				.plain("THEN")
				.ref(ref)
				.plain("ELSE")
				.plain("''")
				.plain("END")
				.build();
	}
}
