/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;

import com.sap.cds.feature.config.Properties;
import com.sap.cds.feature.config.pojo.CdsProperties.Application.ApplicationServiceConfig;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.utils.DraftUtils;

public class DraftServiceConfiguration implements CdsRuntimeConfiguration {

	private AtomicBoolean createdDraftServices = new AtomicBoolean(false);

	@Override
	public void services(CdsRuntimeConfigurer configurer) {
		CdsModel model = configurer.getCdsRuntime().getCdsModel();
		model.services()
		.filter(service -> DraftUtils.isDraftEnabled(service.getQualifiedName(), model))
		.map(service -> service.getQualifiedName())
		.flatMap(name -> {
			List<ApplicationServiceConfig> serviceConfigs = Properties.getCds().getApplication().getServicesByModel(name);
			if(serviceConfigs.isEmpty()) {
				ApplicationServiceConfig config = Properties.getCds().getApplication().getService(name);
				if(config.getModel() == null) {
					return Stream.of(config);
				} else {
					// no service is bound to the model by configuration
					// the service with the same name of the model is bound to a different service
					// => strange configuration, but in that case we don't create any service of that model
					return Stream.empty();
				}
			} else {
				return serviceConfigs.stream();
			}
		})
		.map(config -> new DraftServiceImpl(config, configurer.getCdsRuntime()))
		.peek(config -> createdDraftServices.set(true))
		.forEach(configurer::service);
	}

	@Override
	public void eventHandlers(CdsRuntimeConfigurer configurer) {
		if(createdDraftServices.get()) {
			configurer.eventHandler(new DraftHandler(configurer.getCdsRuntime()));
			configurer.eventHandler(new DraftGCTimerInitializer(configurer.getCdsRuntime()));
		}
	}

}
