/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import java.util.Map;

import com.sap.cds.Result;
import com.sap.cds.feature.config.pojo.CdsProperties.Application.ApplicationServiceConfig;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.services.draft.DraftCancelEventContext;
import com.sap.cds.services.draft.DraftCreateEventContext;
import com.sap.cds.services.draft.DraftEditEventContext;
import com.sap.cds.services.draft.DraftGcEventContext;
import com.sap.cds.services.draft.DraftNewEventContext;
import com.sap.cds.services.draft.DraftPatchEventContext;
import com.sap.cds.services.draft.DraftPrepareEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.impl.cds.ApplicationServiceImpl;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.model.CqnUtils;

public class DraftServiceImpl extends ApplicationServiceImpl implements DraftService {

	public DraftServiceImpl(ApplicationServiceConfig config, CdsRuntime runtime) {
		super(config, runtime);
	}

	@Override
	public Result saveDraft(CqnSelect select, Object... paramValues) {
		return saveDraft(select, CqnUtils.convertToIndexMap(paramValues));
	}

	@Override
	public Result saveDraft(CqnSelect select, Map<String, Object> namedValues) {
		DraftSaveEventContext context = DraftSaveEventContext.create(getTargetEntity(select));

		context.setCqn(select);
		context.setCqnNamedValues(namedValues);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result prepareDraft(CqnSelect select, String sideEffectsQualifier, Object... paramValues) {
		return prepareDraft(select, sideEffectsQualifier, CqnUtils.convertToIndexMap(paramValues));
	}

	@Override
	public Result prepareDraft(CqnSelect select, String sideEffectsQualifier, Map<String, Object> namedValues) {
		DraftPrepareEventContext context = DraftPrepareEventContext.create(getTargetEntity(select));

		context.setCqn(select);
		context.setSideEffectsQualifier(sideEffectsQualifier);
		context.setCqnNamedValues(namedValues);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result editDraft(CqnSelect select, boolean preserveChanges, Object... paramValues) {
		return editDraft(select, preserveChanges, CqnUtils.convertToIndexMap(paramValues));
	}

	@Override
	public Result editDraft(CqnSelect select, boolean preserveChanges, Map<String, Object> namedValues) {
		DraftEditEventContext context = DraftEditEventContext.create(getTargetEntity(select));

		context.setCqn(select);
		context.setPreserveChanges(preserveChanges);
		context.setCqnNamedValues(namedValues);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result newDraft(CqnInsert insert) {
		DraftNewEventContext context = DraftNewEventContext.create(getTargetEntity(insert));

		context.setCqn(insert);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result patchDraft(CqnUpdate update, Object... paramValues) {
		return patchDraft(update, CqnUtils.convertToIndexMap(paramValues));
	}

	@Override
	public Result patchDraft(CqnUpdate update, Map<String, Object> namedValues) {
		return patchDraft(update, list(namedValues));
	}

	@Override
	public Result patchDraft(CqnUpdate update, Iterable<Map<String, Object>> valueSets) {
		DraftPatchEventContext context = DraftPatchEventContext.create(getTargetEntity(update));

		context.setCqn(update);
		context.setCqnValueSets(valueSets);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result cancelDraft(CqnDelete delete, Iterable<Map<String, Object>> valueSets) {
		DraftCancelEventContext context = DraftCancelEventContext.create(getTargetEntity(delete));

		context.setCqn(delete);
		context.setCqnValueSets(valueSets);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result cancelDraft(CqnDelete delete, Object... paramValues) {
		return cancelDraft(delete, CqnUtils.convertToIndexMap(paramValues));
	}

	@Override
	public Result cancelDraft(CqnDelete delete, Map<String, Object> namedValues) {
		return cancelDraft(delete, list(namedValues));
	}

	/**
	 * Creates a draft entity by executing the {@code insert} statement. During execution
	 * of this method, no draft fields will be added. Hence, they must already be
	 * contained in the data of the {@code insert} statement.
	 * @param insert the statement to execute
	 * @return the {@link Result} of the statement
	 */
	public Result createDraft(CqnInsert insert) {
		DraftCreateEventContext context = DraftCreateEventContext.create(getTargetEntity(insert));

		context.setCqn(insert);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result gcDrafts() {
		DraftGcEventContext context = DraftGcEventContext.create();
		emit(context);
		return context.getResult();
	}

}
