/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import static com.sap.cds.services.draft.Drafts.DRAFT_ADMINISTRATIVE_DATA;
import static com.sap.cds.services.impl.draft.CqnAdapter.DRAFT_SUFFIX;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import com.sap.cds.feature.config.Properties;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.RefSegment;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnModifier;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.utils.DraftUtils;

/**
 * A {@link CqnModifier} for the reference of a statement to a draft enabled entity.
 */
public class ReferenceModifier implements CqnModifier {

	public static final String SIBLING  = "SiblingEntity";
	public static final String SIBLING_UNSECURED = SIBLING + "_unsecured";

	private final boolean addSecurityConstraints;
	private boolean forDraft;
	private final boolean modifyIsActiveOnly;
	private final CdsEntity entity;
	private final List<Boolean> followActiveAssociation;
	private final EventContext context;
	private final boolean modifyIsActiveEntity;

	public ReferenceModifier(CdsEntity entity, List<Boolean> followActiveAssociation, boolean addSecurityConstraints, EventContext context) {
		this(entity, false, followActiveAssociation, addSecurityConstraints, context, false);
	}

	public ReferenceModifier(CdsEntity entity, boolean modifyIsActiveOnly, List<Boolean> followActiveAssociation, boolean addSecurityConstraints
			, EventContext context, boolean modifyIsActiveEntity) {
		this.forDraft = entity.getQualifiedName().endsWith(DRAFT_SUFFIX);
		this.entity = entity;
		this.followActiveAssociation = followActiveAssociation; // NOSONAR
		this.modifyIsActiveOnly = modifyIsActiveOnly;
		this.addSecurityConstraints = addSecurityConstraints;
		this.context = context;
		this.modifyIsActiveEntity = modifyIsActiveEntity;
	}

	@Override
	public CqnStructuredTypeRef ref(StructuredTypeRef ref) {
		CdsEntity entity = this.entity;
		List<Segment> newSegments = new ArrayList<>();
		Iterator<Boolean> iterActiveAssoc = followActiveAssociation.iterator();
		boolean isRootEntity = true;
		for (RefSegment segment: ref.segments()) {
			String id = segment.id();
			isRootEntity = isRootEntity && entity.getQualifiedName().equals(id);
			if (SIBLING.equals(id)) {
				forDraft = !forDraft;
			} else if (SIBLING_UNSECURED.equals(id)){
				segment.id(SIBLING);
				forDraft = !forDraft;
			} else if (DRAFT_ADMINISTRATIVE_DATA.equals(id) && !forDraft) {
				forDraft = !forDraft;
				newSegments.add(CQL.refSegment(SIBLING_UNSECURED));
			} else if ((!CdsModelUtils.isAssociationToParentOrChild(id, entity)
					&& entity.findAssociation(id).isPresent()) || (isRootEntity && DraftUtils.isDraftEnabled(entity))) {
				if ((entity.findAssociation(id + DRAFT_SUFFIX).isPresent() || isRootEntity) && iterActiveAssoc.hasNext()) {
					forDraft = !iterActiveAssoc.next();
					if (forDraft && !id.endsWith(DRAFT_SUFFIX)) {
						id += DRAFT_SUFFIX;
						segment.id(id);
						if (addSecurityConstraints) {
							CqnPredicate securityConstraints = CqnAdapter.getSecurityConstraints(context);
							if (securityConstraints != null) {
								if (segment.filter().isPresent()) {
									segment.filter().map(f -> CQL.and(f, securityConstraints)).ifPresent(segment::filter);
								} else {
									segment.filter(securityConstraints);
								}
							}
						}
						if (isRootEntity) {
							// the draft entity
							entity = entity.getTargetOf(SIBLING);
						}
					} else if (id.endsWith(DRAFT_SUFFIX)) {
						id = id.substring(0, id.indexOf(DRAFT_SUFFIX));
						segment.id(id);
						forDraft = false;
						if (isRootEntity) {
							// the active entity
							entity = entity.getTargetOf(SIBLING);
						}
					}
				} else {
					forDraft = false;
				}
			}
			if (!isRootEntity) {
				if (entity.findAssociation(id).isPresent()) {
					entity = entity.getTargetOf(id);
				} else {
					break;
				}
			}
			CdsEntity finalEntity = entity;
			segment.filter().map(f -> CQL.copy(f, new DraftModifier(finalEntity, modifyIsActiveOnly, modifyIsActiveEntity))).ifPresent(segment::filter);
			newSegments.add(segment);
		}
		return CQL.to(newSegments).asRef();
	}

	public boolean getForDraft() {
		return forDraft;
	}

	public static List<List<Boolean>> getAssociationDirections(CqnStatement statement, EventContext context) {
		CqnAnalyzer analyzer = CqnAnalyzer.create(context.getModel());
		AnalysisResult analysisResult;
		if (statement.isSelect()) {
			analysisResult = analyzer.analyze(statement.asSelect());
		} else if (statement.isUpdate()) {
			analysisResult = analyzer.analyze(statement.asUpdate());
		} else {
			analysisResult = analyzer.analyze(statement.asDelete());
		}
		return getAssociationDirections(analysisResult.rootEntity(), statement.ref(), context, analysisResult);
	}

	/**
	 * @param entity the entity
	 * @param ref the reference
	 * @param context the {@link EventContext}
	 * @param analysisResult the {@link AnalysisResult} of the statement. Can also be {@code null}
	 * @return a list of lists that determine whether to follow the draft or active entity
	 */
	public static List<List<Boolean>> getAssociationDirections(CdsEntity entity, CqnStructuredTypeRef ref, EventContext context, AnalysisResult analysisResult) {
		Iterator<ResolvedSegment> iterRes = null;
		if (analysisResult != null) {
			iterRes = analysisResult.iterator();
		}
		List<List<Boolean>> result = new ArrayList<>();
		boolean assocToDraft = Properties.getCds().getDrafts().getAssociationsToInactiveEntities().isEnabled();
		boolean isRootEntity = true;
		Boolean isActiveEntity;
		for (Segment segment: ref.segments()) {
			String id = segment.id();
			isRootEntity = isRootEntity && id.equals(entity.getQualifiedName());
			isActiveEntity = determineIsActiveEntity(iterRes, analysisResult);
			if (!isRootEntity && !assocToDraft && isActiveEntity != null && !isActiveEntity) {
				// if we don't follow associations to draft entities and isActiveEntity = false, the statement will yield no results
				result.clear();
				break;
			}
			if (entity.findAssociation(id + DRAFT_SUFFIX).isPresent() || (isRootEntity && DraftUtils.isDraftEnabled(entity))) {
				if (result.isEmpty()) {
					if (isActiveEntity == null || isActiveEntity) {
						result.add(new ArrayList<>(Arrays.asList(true)));
					}
					if ((assocToDraft || isRootEntity) && (isActiveEntity == null || !isActiveEntity)) {
						result.add(new ArrayList<>(Arrays.asList(false)));
					}
				} else {
					ListIterator<List<Boolean>> iterResult = result.listIterator();
					List<Boolean> tmpList;
					while(iterResult.hasNext()) {
						tmpList = iterResult.next();
						if (isActiveEntity != null) {
							tmpList.add(isActiveEntity);
						} else {
							if (assocToDraft) {
								List<Boolean> newList = new ArrayList<>(tmpList);
								newList.add(false);
								iterResult.add(newList);
							}
							tmpList.add(true);
						}
					}
				}
			}
			if (!isRootEntity) {
				if (entity.findAssociation(id).isPresent()) {
					entity = entity.getTargetOf(id);
				} else {
					break;
				}
			}
		}
		if (result.isEmpty()) {
			result.add(Collections.emptyList());
		}
		return result;
	}

	private static Boolean determineIsActiveEntity(Iterator<ResolvedSegment> iter, AnalysisResult analysisResult) {
		if (iter != null) {
			Boolean result = (Boolean) iter.next().keyValues().get(Drafts.IS_ACTIVE_ENTITY);
			if (result == null && !iter.hasNext()) {
				// last segment -> we also try to get the target keys
				Map<String, Object> targetKeys = analysisResult.targetKeyValues();
				if (targetKeys.get(Drafts.IS_ACTIVE_ENTITY) instanceof Boolean) {
					result = (Boolean) targetKeys.get(Drafts.IS_ACTIVE_ENTITY);
				}
			}
			return result;
		}
		return null;
	}

}
