/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.messages;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;
import java.util.function.Function;

import com.sap.cds.impl.builder.model.StructuredTypeProxy;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.services.messages.MessageTarget;

public class MessageTargetImpl implements MessageTarget {

	public static MessageTarget create(String target) {
		return new MessageTargetImpl(target, null, null);
	}

	public static MessageTarget create(String prefix, String entity, Function<StructuredType<?>, Object> path) {
		return create(prefix, CQL.to(entity), path);
	}

	@SuppressWarnings("unchecked")
	public static <E extends StructuredType<E>> MessageTarget create(String prefix, Class<E> entity, Function<E, Object> path) {
		return create(prefix, StructuredTypeProxy.create(entity), (Function<StructuredType<?>, Object>) path);
	}

	private static MessageTarget create(String prefix, StructuredType<?> entity, Function<StructuredType<?>, Object> path) {
		Object p = path.apply(entity);
		List<Segment> segments = new ArrayList<>();
		if(p instanceof StructuredType) {
			StructuredType<?> type = (StructuredType<?>) p;
			segments.addAll(type.get("SKIP").segments());
			segments.remove(segments.size() - 1);
		} else if(p instanceof ElementRef) {
			segments.addAll(((ElementRef<?>) p).segments());
		} else {
			// invalid specification
			return null;
		}

		Stack<String> entityName = new Stack<>();
		entity.accept(new CqnVisitor() {
			@Override
			public void visit(CqnStructuredTypeRef typeRef) {
				entityName.push(typeRef.firstSegment());
			}
		});

		String name = entityName.pop();
		if(name.equals(segments.get(0).id())) {
			segments.remove(0);
		}

		return new MessageTargetImpl(prefix, name, segments);
	}

	private final String prefix;
	private final List<Segment> path;
	private final String entity;

	private MessageTargetImpl(String prefix, String entity, List<Segment> path) {
		this.prefix = prefix;
		this.entity = entity;
		this.path = path; // NOSONAR
	}

	@Override
	public String getPrefix() {
		return prefix;
	}

	@Override
	public String getEntity() {
		return entity;
	}

	@Override
	public List<Segment> getPath() {
		return path; // NOSONAR
	}

}
