/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.messaging.message;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * This class provides the representation of the S4Hana messages.
 */
public class S4HanaMessage {

	private static final SimpleDateFormat timeDecoder = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssXX");
	private static ObjectMapper mapper = new ObjectMapper();

	/**
	 * Factory method for creating an instance of the S4 HANA message.
	 *
	 * @param event the event name
	 * @param data the structured payload data of the message
	 * @return the {@link S4HanaMessage}
	 */
	public static S4HanaMessage create(String event, Map<String, Object> data) {
		String eventTime;
		synchronized(timeDecoder) {
			eventTime = timeDecoder.format(new Date(System.currentTimeMillis()));
		}

		S4HanaMessage result = new S4HanaMessage(eventTime);
		result.eventType = event;
		result.data = data;

		return result;
	}

	/**
	 * Parses the {@link S4HanaMessage} from the given raw message
	 * @param rawMessage the raw JSON message
	 * @return the {@link S4HanaMessage}
	 */
	public static S4HanaMessage parse(String rawMessage) {
		try {
			S4HanaMessage message = mapper.readValue(rawMessage, S4HanaMessage.class);
			return message;
		} catch (IOException e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_S4_MESSAGE, e);
		}
	}

	@JsonProperty("cloudEventsVersion")
	private String cloudEventsVersion;

	@JsonProperty("schemaURL")
	private String schemaURL;

	@JsonProperty("contentType")
	private String contentType;

	@JsonProperty("source")
	private String source;

	@JsonProperty("eventType")
	private String eventType;

	@JsonProperty("eventID")
	private String eventID;

	@JsonProperty("data")
	private Map<String, Object> data;

	private long eventTime;

	private S4HanaMessage(@JsonProperty("eventTime") String eventTime) {
		if (eventTime != null) {
			try {
				synchronized(timeDecoder) {
					Date date = timeDecoder.parse(eventTime);
					this.eventTime = date.getTime();
				}
			} catch (ParseException e) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_MESSAGE_TIMESTAMP, e);
			}
		}
	}

	/**
	 * Creates the S4HANA event context from the message.
	 *
	 * @return event context
	 */
	public S4HanaMessageEventContext toS4HanaMessageEventContext() {
		S4HanaMessageEventContext ctx;

		ctx = S4HanaMessageEventContext.create(null, eventType);
		ctx.setCloudEventsVersion(cloudEventsVersion);
		ctx.setSchemaURL(schemaURL);
		ctx.setContentType(contentType);
		ctx.setSource(source);
		ctx.setEventType(eventType);
		ctx.setEventID(eventID);
		ctx.setEventTime(eventTime);
		ctx.setData(data);
		return ctx;
	}
}
