/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import java.util.List;
import java.util.Map;

import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = PersistenceService.class)
public class FieldLengthAssertionHandler implements EventHandler {

	@Before(event = { CdsService.EVENT_CREATE, CdsService.EVENT_UPDATE, CdsService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		List<Map<String, Object>> entities = CdsServiceUtils.getEntities(context);

		CdsModelUtils.visitDeep(context.getTarget(), entities, (entity, data, parent) -> {

			// loop over all fields of type CdsSimpleType
			entity.elementsOfType(CdsSimpleType.class)
					// filter out none string fields
					.filter(element -> ((CdsSimpleType) element.getType()).getType() == CdsBaseType.STRING)
					.forEach(element -> {
						validateFieldLength(element, data);
					});
		});
	}

	private static void validateFieldLength(CdsElement element, List<? extends Map<String, Object>> data) {
		Integer maxLength = ((CdsSimpleType) element.getType()).get("length");

		// check if a maximum length is configure in cds model
		if (maxLength != null) {
			String elementName = element.getName();

			for (Map<String, Object> dataEntry : data) {
				String value = (String) dataEntry.get(elementName);

				if (value != null) {
					int valueLength = value.length();
					// check if length of given string exceeds maximum length of field
					if (valueLength > maxLength) {
						throw new ErrorStatusException(CdsErrorStatuses.EXCEEDED_FIELD_LENGTH, value, elementName,
								maxLength);
					}
				}
			}
		}
	}
}
