/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import java.util.List;
import java.util.Map;
import java.util.UUID;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = PersistenceService.class)
public class GenerateUUIDHandler implements EventHandler {

	/**
	 * Make sure, that any entity that is going to be inserted in the given create
	 * event context has a value set in the given key element of type "cds.UUID". If
	 * no value is currently set, a new UUID is generated.
	 *
	 * @param context the create event context
	 */
	@Before(event = { CdsService.EVENT_CREATE, CdsService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.CALCULATE_FIELDS)
	public void fillUUID(EventContext context) {
		// if the entity that is going to be inserted has a key element of type "cds.UUID", generate any missing UUIDs
		List<Map<String, Object>> entries = CdsServiceUtils.getEntities(context);

		// traverse the entity to find all keys
		CdsModelUtils.visitDeep(context.getTarget(), entries, (entity, data, parent) -> {
			// for each
			entity.keyElements().filter(key -> "cds.UUID".equals(key.getType().getQualifiedName()))
			.forEach((key) -> {
				for(Map<String, Object> map : data) {
					if(map.get(key.getName()) == null) {
						// no ID set, generate a guid
						map.put(key.getName(), UUID.randomUUID().toString());
					}
				}
			});
		});
	}

}
