/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;

/**
 * Handler helper class to check not null or mandatory elements.
 */
public abstract class NullElementValidator implements EventHandler {

	@Before(event = { CdsService.EVENT_CREATE, CdsService.EVENT_UPDATE, CdsService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		boolean isPartialUpdate = CdsService.EVENT_UPDATE.equals(context.getEvent());
		// traverse the entity deep
		CdsModelUtils.visitDeep(context.getTarget(), CdsServiceUtils.getEntitiesResolved(context), (entity, data, parent) -> {
			// key elements of the association target are not mandatory, as they get auto-filled
			// returns an empty list, if parent is not set
			List<CdsElement> keyElements = CdsModelUtils.getAssociationKeys(parent);

			// reduce association elements with two representations
			Map<CdsElement, CdsElement> reducedElements = CdsModelUtils.reduceAssociationElements(entity);
			List<Map.Entry<CdsElement, CdsElement>> notNullElements = reducedElements.entrySet().stream()
					// filter all elements, that are keys of the parent association, they are auto-filled by the database
					.filter(entry -> !(keyElements.contains(entry.getKey()) || keyElements.contains(entry.getValue())))
					// filter all not null elements, only the key is relevant here, as for the pair always both elements have the same annotations
					.filter(entry -> checkNull(entry.getKey()))
					.collect(Collectors.toList());

			if(notNullElements.isEmpty()) {
				return;
			}

			// check all not null element names
			for(Map<String, Object> map : data) {
				for(Map.Entry<CdsElement, CdsElement> entry : notNullElements) {
					// if a partial update is allowed, first check if the key (or one of the pair) is set in the data map
					if(isPartialUpdate && !map.containsKey(entry.getKey().getName()) && (entry.getValue() == null || !map.containsKey(entry.getValue().getName()))) {
						continue;
					}

					// if no pair is available only check the key in the map
					// if it is a pair only throw an exception if both elements are null
					if(isNull(map.get(entry.getKey().getName())) && (isNull(entry.getValue()) || isNull(map.get(entry.getValue().getName())))) {
						throw new ErrorStatusException(CdsErrorStatuses.VALUE_REQUIRED,
								entry.getKey().getName() + (entry.getValue() != null ? ("' / '" + entry.getValue().getName()) : ""), entity.getName());
					}
				}
			}
		});
	}

	/**
	 * Returns true, if the element is not allowed to be <code>null</code>
	 * @param element the element to validate
	 * @return true, if the element is not allowed to be <code>null</code>
	 */
	public boolean checkNull(CdsElement element) {
		if(element == null) {
			return false;
		}

		// elements with default values don't have to be checked
		if (element.getType().isSimple() && element.getType().as(CdsSimpleType.class).defaultValue().isPresent()) {
			return false;
		}

		if (element.getType().isAssociation() && element.getName().endsWith("_drafts")) {
			return false;
		}

		return check(element);
	}

	/**
	 * Returns true, if the element is not allowed to be <code>null</code>
	 * @param element the element to validate
	 * @return true, if the element is not allowed to be <code>null</code>
	 */
	protected abstract boolean check(CdsElement element);

	/**
	 * Returns true, if the object is considered to be <code>null</code>
	 * @param obj the object to validate
	 * @return true, if the object is considered to be <code>null</code>
	 */
	protected abstract boolean isNull(Object obj);

}
